#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <stdio.h>
#include <stdint.h>
#include <time.h>

#include "hd44780_test.h"


int main (int argc, char **argv, char **envp)
{
    int device_handle;

    device_handle = open ("/dev/hd44780", O_WRONLY);

    if (device_handle < 0)
    {
        printf ("Kann \"/dev/hd44780\" nicht oeffnen.\n");
        return -1;
    }

    // Pre-Initialisierung des HD44780.
    hd44780_pre_init (device_handle, HD44780_DATA_4_BIT);

    printf ("Cursor OFF\n");
    hd44780_show_cursor (device_handle, OFF);
    printf ("Cursor ON\n");
    hd44780_show_cursor (device_handle, ON);
    printf ("Cursor OFF\n");
    hd44780_show_cursor (device_handle, OFF);
    printf ("Cursor ON\n");
    hd44780_show_cursor (device_handle, ON);

    hd44780_cursor_home (device_handle);
    hd44780_set_line (device_handle, 1);
    hd44780_set_text (device_handle, "Zeile 1");
    hd44780_cursor_home (device_handle);
    hd44780_set_line (device_handle, 2);
    hd44780_set_text (device_handle, "    Zeile 2");
    hd44780_cursor_home (device_handle);
    hd44780_set_line (device_handle, 3);
    hd44780_set_text (device_handle, "         Zeile 3,");
    hd44780_cursor_home (device_handle);
    hd44780_set_line (device_handle, 4);
    hd44780_set_text (device_handle, "      Zeile 4");

    return 0;

    sleep (2);
    hd44780_clear_display (device_handle);
    hd44780_set_line (device_handle, 4);
    sleep (1);
    hd44780_set_text (device_handle, "Alles klar.");
    sleep (2);
    hd44780_cursor_home (device_handle);
    hd44780_set_text (device_handle, "???Zeile 1???");

//    hd44780_clear_display (device_handle);

    close (device_handle);

    return 0;
}

/**
 * hd44780_pre_init
 * ================
 * Einstellung, ob der HD44780 im 4-Bit- oder im 8-Bit-Modus
 * betrieben wird und weitere Grundeinstellungen.
 */
int  hd44780_pre_init (int device_handle, uint8_t mode)
{
    int err;

    cmdbuf [0] = SET_DATA_WIDTH;
    cmdbuf [1] = (unsigned char) mode;

    // Fehlerbehandlung fuer unzulaessiges Argument fehlt.

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_clear_display
 * =====================
 * Loescht den Bildschirminhalt und setzt den Cursor auf die Startposition.
 */
int hd44780_clear_display (int device_handle)
{
    int err;

    cmdbuf [0] = CLEAR_DISPLAY;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_show_cursor
 * ===================
 * Schaltet den Cursor ein (mode = ON) bzw. aus (mode = OFF).
 */
int hd44780_show_cursor (int device_handle, uint8_t mode)
{
    int err;

    cmdbuf [0] = CURSOR_ON_OFF;
    cmdbuf [1] = (uint8_t) mode;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_cursor_home
 * ===================
 * Setzt den Cursor in die Home-Position des Displays. Der derzeit angezeigte
 * Text bleibt erhalten.
 */
int hd44780_cursor_home (int device_handle)
{
    int err;

    cmdbuf [0] = CURSOR_HOME;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_cursor_blink
 * ====================
 * Versetzt den Cursor in den Blink-Modus. Wurde der Cursor vorher nicht
 * angezeigt, so wird die Anzeige des Cursors eingeschaltet.
 */
int hd44780_cursor_blink (int device_handle, uint8_t mode)
{
    int err;

    cmdbuf [0] = CURSOR_BLINK_MODE;
    cmdbuf [1] = (unsigned char) mode;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_set_line
 * ================
 * Bestimmt die Zeile, in der der nachfolgende Text angezeigt werden soll.
 */
int hd44780_set_line (int device_handle, uint8_t line)
{
    int err;

    cmdbuf [0] = SET_LINE_NUM;
    cmdbuf [1] = (uint8_t) line;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_ticker_mode
 * ===================
 * Aktiviert/Deaktiviert die Betriebsart eines "Nachrichtentickers".
 */
int hd44780_ticker_mode (int device_handle, uint8_t mode)
{
    int err;

    cmdbuf [0] = SET_TICKER_MODE;
    cmdbuf [1] = mode;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf)/sizeof (uint8_t));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_set_ticker_dir
 * ======================
 * Dient zur Angabe der Laufrichtung des Tickertextes.
 */
int hd44780_set_ticker_dir (int device_handle, uint8_t dir)
{
    int err;

    cmdbuf [0] = SET_TICKER_DIR;
    cmdbuf [1] = dir;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf)/sizeof (uint8_t));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_text_alignment
 * ======================
 * Legt die Ausrichtung des Textes auf linksbuendig, rechtsbuendig oder
 * zentriert fest.
 */
int hd44780_text_alignment (int device_handle, uint8_t alignment)
{
    int err;

    cmdbuf [0] = SET_TEXT_ALIGN;
    cmdbuf [1] = alignment;

    err = write (device_handle, cmdbuf, sizeof (cmdbuf)/sizeof (uint8_t));
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * hd44780_set_text
 * ================
 * Gibt den angegebenen Text in der vorher festgelegten Zeile aus. Die Zeile
 * wird zunaechst mit Leerzeichen ueberschrieben. Die maximale Textlaenge wird
 * durch die Konstante HD44780_MAX_CHAR_PER_LINE begrenzt.
 */
int hd44780_set_text (int device_handle, char *text)
{
    int err;
    int len;

    cmdbuf [0] = WRITE_TEXT;

    len = strlen (text);
    if (len > MAX_CHARS_PER_LINE)
    {
        len = MAX_CHARS_PER_LINE;
    }

    // Text nach txtbuf kopieren.
    memcpy (txtbuf, text, len);

    err = write (device_handle, txtbuf, len);
    if (err == -1)
    {
        error_message (err, cmdbuf);

        return -HD44780_INVALID_COMMAND;
    }

    return 0;
}

/**
 * error_message
 * =============
 * Fuer alle Funktionen gemeinsame Fehlerausgabe.
 */
void error_message (int err, unsigned char *cmdbuf)
{
    printf ("Invalid command. err = %d.\n", err);
    printf ("Command = %d  :  Argument = %d\n", cmdbuf [0], cmdbuf [1]);

    return;
}
