#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

#include "sn74hc595.h"

/**
 * driver_open
 * ===========
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn dort die
 * open()-Funktion verwendet wird.
 */
static int driver_open (struct inode *device_file, struct file *instance)
{
    int err;
    int gpio_num;

    for (gpio_num = GPIO_OFFSET; gpio_num < GPIO_OFFSET + GPIOS_USED; gpio_num++)
    {
        printk (KERN_INFO "Anforderung von GPIO_%d\n", gpio_num);

        // Anforderung der GPIOs 23 bis 26.
        err = gpio_request(gpio_num, "GPIO");
        if (err)
        {
            printk (KERN_ALERT "gpio_request() failed for GPIO_%d\n", gpio_num);
            return -1;
        }

        // Alle GPIOs werden als Ausgang genutzt.
        err = gpio_direction_output(gpio_num, 0);
        if (err)
        {
            printk (KERN_ALERT "gpio_direction_output() failed for GPIO_%d\n", gpio_num);
            return -1;
        }

        // Meldungen
        printk (KERN_INFO "GPIO_%d bereit\n", gpio_num);
    }

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * driver_close
 * ============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn das
 * Userspace-Programm beendet wird.
 */
static int driver_close (struct inode *device_file, struct file *instance)
{
    int gpio_num;

    gpio_set_value(GPIO23_SERDATA, 0);
    gpio_set_value(GPIO24_RCLK, 0);
    gpio_set_value(GPIO25_SRCLK, 0);

    for (gpio_num = GPIO_OFFSET; gpio_num < GPIO_OFFSET + GPIOS_USED; gpio_num++)
    {
        printk (KERN_INFO "Gebe GPIO_%d frei.\n", gpio_num);
        gpio_free(gpio_num);
    }

    return 0;
}

/**
 * driver_write
 * ============
 * Diese Funktion wird aus dem Userspace aufgerufen, z.B. bei
 *    echo "text" > /dev/treiber.
 */
static ssize_t driver_write (struct file *instance, const char __user *user,
                             size_t count, loff_t *offset)
{
    unsigned long not_copied, to_copy;
    u32           value;
    u8            shift_count = 0;

    to_copy = min (count, sizeof (value));
    not_copied = copy_from_user(&value, user, to_copy);

    gpio_set_value(GPIO24_RCLK, LOW);           // Latch-Takt -> LOW
    gpio_set_value(GPIO25_SRCLK, LOW);          // Ser. Takt  -> LOW

    /**
     * Zahl der Schleifendurchlaeufe (Bits) ist abhaengig von der
     * der Schieberegister (SN74HC595).
     */
    for (shift_count = 0; shift_count < NUM_SN74HC595 * 8; shift_count++)
    {
        gpio_set_value(GPIO23_SERDATA, value);  // Datenbit -> GPIO23
        gpio_set_value(GPIO25_SRCLK, HIGH);     // Ser. Takt LOW -> HIGH
        value <<= 1;                            // Naechstes Bit
        msleep(5);                              // Warte 5ms
        gpio_set_value(GPIO25_SRCLK, LOW);      // Ser. Takt HIGH -> LOW

        gpio_set_value(GPIO24_RCLK, HIGH);      // Latch-Takt LOW -> HIGH
        msleep(5);                              // Warte 5ms
        gpio_set_value(GPIO24_RCLK, LOW);       // Latch-Takt HIGH -> LOW
    }

    gpio_set_value(GPIO24_RCLK, LOW);           // Latch-Takt -> LOW
    gpio_set_value(GPIO25_SRCLK, LOW);          // Ser. Takt  -> LOW

    return to_copy - not_copied;
}

/**
 * struct file_operations
 */
static struct file_operations fops =
{
    owner   : THIS_MODULE,
    open    : driver_open,
    release : driver_close,
    write   : driver_write
};


/**
 * mod_init
 * ========
 * Wird beim Laden des Treibers/Moduls aufgerufen, z.B. bei insmod, modprobe.
 */
static int __init mod_init (void)
{
    if (alloc_chrdev_region(&hc595_dev_num, 0, 1, DEVICE_NAME) < 0)
    {
        printk (KERN_ALERT "alloc_chrdev_region failed with MAJOR = %d\n", MAJOR(hc595_dev_num));
        return -EIO;
    }

    hc595_object = cdev_alloc();
    if (hc595_object == NULL)
    {
        goto free_device_number;
    }

    hc595_object->owner = THIS_MODULE;
    hc595_object->ops   = &fops;

    if (cdev_add (hc595_object, hc595_dev_num, 1))
    {
        goto free_cdev;
    }

    hc595_class = class_create(THIS_MODULE, DEVICE_NAME);

    if (IS_ERR(hc595_class))
    {
        pr_err("%s: no udev support\n", DEVICE_NAME);
        goto free_cdev;
    }

    hc595_dev = device_create(hc595_class, NULL, hc595_dev_num, NULL, "%s", DEVICE_NAME);
    dev_info (hc595_dev, "mod_init_called\n");
    return 0;

free_cdev:
    kobject_put(&hc595_object->kobj);

free_device_number:
    unregister_chrdev_region(hc595_dev_num, 1);
    printk("mod_init failed\n");
    return -EIO;
}

/**
 * mod_exit
 * ========
 * Wird aufgerufen, wenn Treiber/Modul entladen werden.
 */
static void __exit mod_exit (void)
{
    dev_info(hc595_dev, "mod_exit called\n");
    device_destroy(hc595_class, hc595_dev_num);
    class_destroy(hc595_class);
    cdev_del(hc595_object);
    unregister_chrdev_region(hc595_dev_num, 1);
    return;
}


/**
 * Zusatzinformationen, teilweise vorgeschrieben.
 */
module_init (mod_init);
module_exit (mod_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR ("Ralf Jesse");
MODULE_DESCRIPTION("Kernelmodul zur Ansteuerung eines Schieberegisters vom Typ SN74HC595.");
