#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <asm/uaccess.h>

#include "gpio_pi.h"
#define GPIO_23     23

static int driver_open (struct inode *device_file, struct file *instance)
{
    int err;

    err = gpio_request(GPIO_23, "rpi-gpio-23");
    printk(KERN_ALERT "driver_open. err = %d\n", err);

    if (err)
    {
        printk(KERN_ALERT "gpio-request failed %d\n", err);
        return -1;
    }

    err = gpio_direction_output(23,0);

    if (err)
    {
        printk (KERN_ALERT "gpio_direction_output failed %d\n", err);
        return -1;
    }

    printk (KERN_ALERT "gpio 23 successfully configured\n");
    return 0;
}

static int driver_close (struct inode *device_file, struct file *instance)
{
    printk (KERN_ALERT "driver_close called\n");
    gpio_free(23);

    return 0;
}

static ssize_t driver_write (struct file *instance, const char __user *user,
                             size_t count, loff_t *offset)
{
    unsigned long not_copied, to_copy;
    u32           value = 0;

    to_copy = min (count, sizeof (value));
    not_copied = copy_from_user(&value, user, to_copy);
    printk (KERN_ALERT "driver_write (%d)\n", value);

    if (0 == value)
    {
        gpio_set_value(23, 0);
    }
    else
    {
        gpio_set_value(23, 1);
    }
    return to_copy - not_copied;
}

static int __init mod_init (void)
{
    if (alloc_chrdev_region(&fastgpio_dev_num, 0, 1, "gpio_pi") < 0)
    {
        printk (KERN_ALERT "alloc_chrdev_region failed with MAJOR = %d\n", MAJOR(fastgpio_dev_num));
        return -EIO;
    }

    driver_object = cdev_alloc();

    if (driver_object == NULL)
    {
        goto free_device_number;
    }

    driver_object->owner = THIS_MODULE;
    driver_object->ops   = &fops;

    if (cdev_add (driver_object, fastgpio_dev_num, 1))
    {
        goto free_cdev;
    }

    fastgpio_class = class_create(THIS_MODULE, "gpio_pi");

    if (IS_ERR( fastgpio_class))
    {
        pr_err("gpio_pi: no udev support\n");
        goto free_cdev;
    }

    fastgpio_dev = device_create(fastgpio_class, NULL, fastgpio_dev_num, NULL, "%s", "gpio_pi");
    dev_info (fastgpio_dev, "mod_init_called\n");
    return 0;

free_cdev:
    kobject_put(&driver_object->kobj);

free_device_number:
    unregister_chrdev_region(fastgpio_dev_num, 1);
    printk("mod_init failed\n");
    return -EIO;
}

static void __exit mod_exit (void)
{
    dev_info( fastgpio_dev, "mod_exit called\n");
    device_destroy(fastgpio_class, fastgpio_dev_num);
    class_destroy(fastgpio_class);
    cdev_del(driver_object);
    unregister_chrdev_region(fastgpio_dev_num, 1);
    return;
}

module_init (mod_init);
module_exit (mod_exit);
MODULE_LICENSE("GPL");
