#ifndef MAX7219_H_INCLUDED
#define MAX7219_H_INCLUDED


/**
 * Standardfunktionen/-register des MAX7219
 */
#define MAX7219_NOOP        0x00
#define MAX7219_DIG0        0x01
#define MAX7219_DIG1        0x02
#define MAX7219_DIG2        0x03
#define MAX7219_DIG3        0x04
#define MAX7219_DIG4        0x05
#define MAX7219_DIG5        0x06
#define MAX7219_DIG6        0x07
#define MAX7219_DIG7        0x08
#define MAX7219_DECODEMODE  0x09
#define MAX7219_INTENSITY   0x0A
#define MAX7219_SCANLIMIT   0x0B
#define MAX7219_OPMODE      0x0C
#define MAX7219_DISPTEST    0x0F

/**
 * Sonderfunktionen zur Steuerung des MAX7219. Sie belegen das obere Nibble des
 * Kommandobytes. Die MAX7219 ignorieren dieses Nibble, der Treiber wertet sie
 * aber aus.
 */
#define MAX7219_SET_NUM     0x10
#define MAX7219_SHOW_BITMAP 0x11

/**
 * Betriebsarten
 */
#define OPMODE_SHUTDOWN 0x00
#define OPMODE_NORMAL   0x01;
#define OFF             0x00;
#define ON              0x01;

/**
 * Bereich der zulaessigen Intensitaet.
 */
#define MAX7219_MIN_INTENSITY   0x00    // Minimale Helligkeit der LEDs
#define MAX7219_MAX_INTENSITY   0x0F    // Maximale Helligkeit der LEDs
#define MAX7219_MED_INTENSITY   0x07    // Mittlere Helligkeit der LEDs

// Allgemeine Definitionen
#define GPIOS_USED       3
#define GPIO_OFFSET     23
#define GPIO23_DIN      23  // Pin 16 (Broadcom-Belegung)
#define GPIO24_LOAD     24  // Pin 18 (Broadcom-Belegung)
#define GPIO25_CLK      25  // Pin 22 (Broadcom-Belegung)

#define DEVICE_NAME     "max7219"
#define LOW             0
#define HIGH            1
#define NUM_MAX7219     4   // Anzahl der MAX7219-Bausteine
#define NUM_COLUMNS     8   // Anzahl der Spalten der Matrix
#define MAX_BITPOS     16   // (pos + 1) des hoechstwertigen Bits eines Wertes

// Treiberglobale Variablen
static        dev_t   max7219_majmin;       // Major-/Minor-Nummer
static struct cdev   *max7219_object;
static struct class  *max7219_class;
static struct device *max7219_dev;
static        uint8_t max7219_count = 1;    // Anzahl der MAX7219-Chips

// Daten aus dem Userspace werden in Werten vom Typ uint8_t uebertragen. Die
// Bits werden in umgekehrter Reihenfolge beginnend mit Bit 7 an die einzelnen
// MAX7219 uebertragen. Daher beginnen wir mit bitPos = 8;
uint16_t      bitPos = 8;

// Ein MAX7219 kann bis zu acht Spalten ansteuern. Gesamtzahl der Spalten
// ist demnach 8 * NUM_MAX7219.
uint8_t       buf [8 * NUM_MAX7219];


// Funktionsprototypen f. Kernfunktionen des Treibers
static int  __init  max7219_init  (void);
static void __exit  max7219_exit  (void);
static int          max7219_open  (struct inode *device_file,
                                   struct file *instance);
static int          max7219_close (struct inode *device_file,
                                   struct file *instance);
static ssize_t      max7219_write (struct file *instance,
                                   const char __user *user,
                                   size_t count,
                                   loff_t *offset);

// Funktionsprototypen fuer Komfortfunktionen des Treibers
static void max7219_set_num_devices (uint8_t);
static void max7219_set_op_mode (uint8_t, uint8_t);
static void max7219_set_decode_mode (uint8_t, uint8_t);
static void max7219_set_intensity (uint8_t, uint8_t);
static void max7219_set_scan_limit (uint8_t, uint8_t);
static void max7219_exec_disp_test (uint8_t, uint8_t);
static void max7219_clock_out_bit_data (uint16_t);
static void max7219_show_bitmap (uint8_t, uint8_t);

// Funktionsprototypen fuer Hilfsfunktionen
static void         printBin (uint16_t);
static void         max7219_send_noop (void);
static void         max7219_make_data (uint8_t, uint16_t);

#endif // MAX7219_H_INCLUDED
