#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <linux/time.h>

#include "max7219imp.h"

/**
 * max7219_open
 * ============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn dort die
 * open()-Funktion verwendet wird.
 */
static int max7219_open (struct inode *device_file, struct file *instance)
{
    int err;
    int gpio_num;

    for (gpio_num = GPIO_OFFSET;
         gpio_num < GPIO_OFFSET + GPIOS_USED;
         gpio_num++)
    {
        // Anforderung der benoetigten GPIOs.
        err = gpio_request(gpio_num, "GPIO");
        if (err)
        {
            pr_err("%s: gpio_request() failed.\n", DEVICE_NAME);
            return -1;
        }

        // Alle GPIOs werden als Ausgang genutzt.
        err = gpio_direction_output(gpio_num, 0);
        if (err)
        {
            pr_err ("%s: gpio_direction_output() failed.\n", DEVICE_NAME);
            return -1;
        }
    }

    // Zu Beginn werden die Startpegel aller verwendeten Ports gesetzt.
    gpio_set_value(GPIO23_DIN, LOW);
    gpio_set_value(GPIO24_LOAD, HIGH);
    gpio_set_value(GPIO25_CLK, LOW);

    dev_info (max7219_dev, "max7219_open called\n");

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * max7219_close
 * =============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn das
 * Userspace-Programm beendet wird.
 */
static int max7219_close (struct inode *device_file, struct file *instance)
{
    int gpio_num;

    max7219_shutDown ();

    gpio_set_value(GPIO23_DIN, LOW);
    gpio_set_value(GPIO24_LOAD, LOW);
    gpio_set_value(GPIO25_CLK, LOW);

    for (gpio_num = GPIO_OFFSET;
         gpio_num < GPIO_OFFSET + GPIOS_USED;
         gpio_num++)
    {
        gpio_free(gpio_num);
    }
    dev_info (max7219_dev, "max7219_close called\n");

    return 0;
}

/**
 * max7219_write
 * =============
 * Diese Funktion wird aus dem Userspace aufgerufen, z.B. bei
 * echo "text" > /dev/treiber.
 */
static ssize_t max7219_write ( struct file *instance,
                               const char __user *user,
                               size_t count,
                               loff_t *offset
                             )
{
    unsigned long not_copied, to_copy;
    uint16_t      val;          // Wert aus der Applikation im Userspace
    uint16_t      bitPos = 16;  // Wert hat eine Laenge von 16 Bit
    uint16_t      sendBit;      // Es wird immer nur ein Bit gesendet
    uint16_t      mask;         // Zaehlt die Anzahl der Schiebeoperationen

    to_copy = min (count, sizeof (val));
    not_copied = copy_from_user (&val, user, to_copy);

    // Vor jeder Datenuebertragung wird LOAD auf LOW gesetzt. Die letzten 16 Bit
    // werden dann mit der steigenden Flanke von LOAD uebernommen.
    gpio_set_value(GPIO24_LOAD, LOW);

    // Zuerst wird Bit 15 uebertragen, dann Bit 14 usw. bis Bit 0 am Ende. Die
    // Bits liegen somit in der umgekehrten Reihenfolge vor!
    while (bitPos > 0)
    {
        mask = 1 << (bitPos -1); // Make fuer das naechste Bit ermitteln ...
        sendBit = val & mask;    // ... und anwenden.

        // Sende den Wert entsprechend der Maske an GPIO23_DIN
        gpio_set_value (GPIO23_DIN, ((sendBit) ? HIGH : LOW));

        // Taktimpuls erzeugen. Laut Datenblatt muessen LOW und HIGH jeweils
        // mindestens 50 ns anstehen --> Hier gewaehlt: 1 µs
        gpio_set_value (GPIO25_CLK, HIGH);
        udelay (1);
        gpio_set_value (GPIO25_CLK, LOW);
        udelay (1);

        --bitPos;  // Bitposition fuer das naechste Bit bestimmen.
    }

    // Uebernahme der letzten empfangenen 16 Bit --> GPIO24_LOAD = HIGH setzen
    gpio_set_value (GPIO24_LOAD, HIGH);

    return to_copy - not_copied;
}

/**
 * struct file_operations
 */
static struct file_operations fops =
{
    owner   : THIS_MODULE,
    open    : max7219_open,
    release : max7219_close,
    write   : max7219_write
};


/**
 * max7219_init
 * ============
 * Wird beim Laden des Treibers/Moduls aufgerufen, z.B. bei insmod, modprobe.
 */
static int __init max7219_init (void)
{
    if (alloc_chrdev_region(&max7219_numdevs, 0, 1, DEVICE_NAME) < 0)
    {
        pr_err ("%s: alloc_chrdev_region() failed.\n", DEVICE_NAME);
        return -EIO;
    }

    max7219_object = cdev_alloc();
    if (max7219_object == NULL)
    {
        goto free_device_number;
    }

    max7219_object->owner = THIS_MODULE;
    max7219_object->ops   = &fops;

    if (cdev_add (max7219_object, max7219_numdevs, 1))
    {
        goto free_cdev;
    }

    max7219_class = class_create(THIS_MODULE, DEVICE_NAME);

    if (IS_ERR(max7219_class))
    {
        pr_err("%s: no udev support\n", DEVICE_NAME);
        goto free_cdev;
    }

    max7219_dev = device_create(max7219_class, NULL, max7219_numdevs, NULL,
                                "%s", DEVICE_NAME);
    dev_info (max7219_dev, "max7219_init_called\n");

    return 0;

free_cdev:
    kobject_put(&max7219_object->kobj);

free_device_number:
    unregister_chrdev_region(max7219_numdevs, 1);
    return -EIO;
}

/**
 * max7219_exit
 * ============
 * Wird aufgerufen, wenn Treiber/Modul entladen werden.
 */
static void __exit max7219_exit (void)
{
    max7219_shutDown();
    dev_info(max7219_dev, "max7219_exit called\n");
    device_destroy(max7219_class, max7219_numdevs);
    class_destroy(max7219_class);
    cdev_del(max7219_object);
    unregister_chrdev_region(max7219_numdevs, 1);
    return;
}

/**
 * max7219_shutDown
 * ================
 * Wird aufgerufen, wenn Treiber/Modul entladen werden.
 */
static void max7219_shutDown (void)
{
    uint16_t val = 0x0C00;
    uint16_t bitPos = 16;
    uint16_t sendBit;
    uint16_t mask;

    gpio_set_value(GPIO24_LOAD, HIGH);
    while (bitPos > 0)
    {
        mask = 0x0001 << (bitPos - 1);
        sendBit = val & mask;

        // Sende den Wert entsprechend der Maske an GPIO23_DIN
        gpio_set_value (GPIO23_DIN, ((sendBit) ? HIGH : LOW));

        // Taktsignal erzeugen
        gpio_set_value (GPIO25_CLK, HIGH);
        udelay (1);
        gpio_set_value (GPIO25_CLK, LOW);
        udelay (1);

        --bitPos;   // Maske des naechsten Bits vorbereiten
    }
    gpio_set_value(GPIO24_LOAD, LOW);
}


/**
 * Zusatzinformationen, teilweise vorgeschrieben.
 */
module_init (max7219_init);
module_exit (max7219_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR ("Ralf Jesse");
MODULE_DESCRIPTION("Kernelmodul zur Ansteuerung eines Schieberegisters vom Typ MAXIM 7219.");
