#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

#include "mydevice.h"

int __init mydevice_init (void)
{
    // Anforderung einer Geraetenummer
    if (alloc_chrdev_region (&mydevice_devNum, 0, 1, "mydevice_") < 0)
    {
        return -EIO;
    }

    // Anforderung einer cdev-Struktur
    mydevice_object = cdev_alloc ();
    if (NULL == mydevice_object)    // Erfolg pruefen
    {
        goto release_mydev_devNum;
    }

    // Notwendige Informationen in cdev-Struktur eintragen
    mydevice_object->owner = THIS_MODULE;
    mydevice_object->ops  = &fops;

    // Zeichenorientiertes Geraet hinzufuegen
    if (cdev_add (mydevice_object, mydevice_devNum, 1))
    {
        goto release_cdev;
    }

    // Das Modul in /sys/module eintragen
    mydev_class = class_create (THIS_MODULE, "mydevice_");
    if (IS_ERR (mydevice_class))  // Erfolg ueberpruefen
    {
        goto release_cdev;
    }
    // Geraetedatei /dev/mydevice erzeugen
    mydevice_device = device_create (mydevice_class, NULL, mydevice_devNum,
                                     "%s", "mydevice_");

    return 0;  // Es hat alles funktioniert

// Wenn Fehler bei einem der vorherigen Schritte aufgetreten sind
release_cdev:
    /**
     * Dekrementiert Referenzcounter fuer den Treiber und ruft cleanup() auf,
     * wenn der counter = 0 ist.
     */
    kobject_put (&mydevice_object->kobj);

release_mydevice_number:
    /**
     * Freigabe der reservierten Major-/Minor-Nummer(n)
     */
    unregister_chrdev_region (mydevice_devNum, 1);
    return -EIO;
}

/**
 * void __exit mydevice_exit (void)
 * ===============================
 * Entfernt ein Gearaet aus dem System.
 *
 * Verwendete Funktionen:
 * device_destroy : Entfernt ein Geraet, das mit device_create erzeugt wurde.
 * class_destroy  : Entfernt ein Geraet, das mit class_create erzeugt wurde.
 * cdev_del       : Entfernt die angegebene Geraetestruktur. Moeglicherweise
 *                  wird sogar die Struktur selber freigegeben (dies ist nicht
 *                  garantiert).
 */
void __exit mydevice_exit (void)
{
    // mydevice-Geraetedatei entladen
    device_destroy (mydevice_class, mydevice_devNum);
    class_destroy  (mydevice_class);
    cdev_del (mydevice_object);
    unregister_chrdev (mydevice_devNum, DEVICE_NAME);
    printk (KERN_INFO "Geraet /dev/%s wurde entfernt.\n", DEVICE_NAME);
}

/**
 * mydevice_open
 * ============
 * Wird aufgerufen, wenn eine Anwendung aus dem Userspace das Geraet benutzen
 * moechte.
 *
 * Parameter:
 * inode_p : Pointer auf den Inode des Geraets
 * file_p  : Poiter auf die Geraetebezeichnung
 *
 * Returns:
 * int          : Fehlermeldung oder 0 im Erfolgsfall
 */
static int mydevice_open (struct inode *inode_p, struct file *file_p)
{
    static int counter = 0;

    // Wenn das Geraet bereits "geoeffnet" ist -> Busy
    if (device_open)
    {
        return -EBUSY;
    }

    device_open++;    // Sperre gegen nochmaliges Oeffnen der Geraetedatei

    return SUCCESS;
}

/**
 * mydevice_close
* ===============
 * Wird aufgerufen, wenn ein Prozess das Gearaet schliesst. Die Parameter und
 * Rueckgabewerte entsprechen der open-Funktion.
 */
static int mydevice_close (struct inode *inode_p, struct file *file_p)
{
    device_open--;    // Damit das Geraet erneut genutzt werden kann.

    return 0;
}

/**
 * mydevice_read
 * ============
 * Lesen von der Geraetedatei
 *
 * Parameter:
 * file_p   : Pointer auf struct file
 * *buffer  : Buffer mit den anzuzeigenden Daten
 * length   : Laenge des Buffers
 * offset_p :
 */

/**
 * Wird aufgerufen, wenn ein Prozess aus dem Geraet /dev/mydevice liest.
*/
static ssize_t mydevice_read (struct file *file_p,
                             char        *buffer,
                             size_t       length,
                             loff_t      *offset_p)
{
    int           bytes_read = 0;
    unsigned long to_copy;     // Anzahl der zu kopierenden Daten
    unsigned long not_copied;  // Anzahl der nicht-kopierten Daten
    u32           value = 0;   // Ziel der kopierten Daten

    // HIER FINDET DIE WIRKLICHE »ACTION« STATT, Z.B. EINLESEN VON SENSOR-
    // DATEN.

    // Zunaechst wird ermittelt, wieviele Daten gelesen werden sollen.
    // Hierfür wird die min()-Funktion verwendet.
    to_copy    = min (count, sizeof (value));
    not_copied = copy_to_user (user, &value, to_copy);

    // Rueckgabe der nicht-kopierten (!!!) Daten
    return to_copy - not_copied;
}

/**
 * Wird aufgerufen, wenn ein Prozess in das Geraet /dev/mydevice schreibt.
*/
static ssize_t mydevice_write(struct file *file_p, const char *buff, size_t len,
                              loff_t *off_p)
{
    unsigned long not_copied;
    unsigned long to_copy;
    u32           value = 0;

    // Ermittlung der Zahl der zu uebertragenden Daten
    to_copy = min (count, sizeof (value));
    not_copied = copy_from_user (&value, user, to_copy);

    // HIER FINDET DIE WIRKLICHE ACTION STATT, Z.B. ANSTEUERUN EINER LED ODER
    // EINES ODER MEHREREN ANDEREN AKTOREN.

    return to_copy - not_copied;
}

module_init (mydevice_init);
module_exit (mydevice_exit);

MODULE_LICENSE ("GPL");
MODULE_AUTHOR  ("Modified and extended by: Ralf Jesse");
