/**
 * main.c
 *
 *  Created on: 25.10.2013
 *      Author: ralf
 *
 * \brief
 */

static void InitAllPios(void);

// Allgemeine Header-Dateien
#include <stdio.h>
#include <stdbool.h>

// Spezielle Header-Dateien fr das Entwicklungsboard und den Chip
#include "board.h"
#include "board_olimex.h"
#include "chip.h"
#include "global.h"
#include "spi_main.h"
#include "sdcard.h"
#include "calc_crc.h"

// Header-Dateien fuer FAT
#include "ff.h"
#include "FAT_EvalReturnCodes.h"


// Diese sind in "board.h" definiert
const Pin pinSD_CD[] =
{ PIN_SD_CD };
const Pin pinSD_WP[] =
{ PIN_SD_WP };
const Pin pinSD_CS[] =
{ PIN_SD_CS };
const Pin pinSD_SCK[] =
{ PIN_SD_SCK };
const Pin pinSD_MOSI[] =
{ PIN_SD_MOSI };
const Pin pinSD_MISO[] =
{ PIN_SD_MISO };

uint32_t timeStamp;
bool sdCardStartupTimeDone = false;
bool sdCardNotUsable = false;

// SD-Karten-Kommandos
unsigned char CMD00[] =
{ 0x40, 0x00, 0x00, 0x00, 0x00 };  // CMD0
unsigned char CMD08[] =
{ 0x48, 0x00, 0x00, 0x01, 0xAA };  // CMD8
unsigned char CMD09[] =
{ 0x49, 0x00, 0x00, 0x00, 0x00 };  // CMD9  --> CSD
unsigned char CMD10[] =
{ 0x4A, 0x00, 0x00, 0x00, 0x00 };  // CMD10 --> CID
unsigned char CMD16[] =
{ 0x50, 0x00, 0x00, 0x00, 0x00 }; // CMD16 --> SET_BLOCKLEN (Default = 512 Byte)
unsigned char CMD17[] =
{ 0x51, 0x00, 0x00, 0x00, 0x00 };  // CMD17 --> READ_SINGLE_BLOCK
unsigned char CMD24[] =
{ 0x58, 0x00, 0x00, 0x00, 0x00 };  // CMD24 --> WRITE_BLOCK
unsigned char CMD41_V1[] =
{ 0x69, 0x00, 0x00, 0x00, 0x00 };  // CMD41 (V1.x-Version), HCS = '0'
unsigned char CMD41_V2[] =
{ 0x69, 0x40, 0x00, 0x00, 0x00 };  // CMD41 (V2.x-Version), HCS = '1'
unsigned char CMD55[] =
{ 0x77, 0x00, 0x00, 0x00, 0x00 };  // CMD55
unsigned char CMD58[] =
{ 0x7A, 0x00, 0x00, 0x00, 0x00 };  // CMD58
unsigned char CMD_DUMMY = 0xFF;

SDINFO_t sdInfo;
CID_t cidInfo;
CSD_t csdInfo;

bool sdCardDetected = false;

// Erforderlich fuer das FAT-Dateisystem
FRESULT result;
FATFS   fs;


/**
 * M A I N - Funktion
 */
int main(void)
{
    bool     msgNoCardFound = false;
    bool     msgCardInitStarted = false;
    bool     msgCardInfoShown = false;


    // Watchdog-Timer deaktivieren
    WDT_Disable(WDT);

    // Erzeugung eines 1ms-Taktes aus dem Systemtakt
    if (SysTick_Config(BOARD_MCK / 1000))
    {
        return -1;
    }

    LED_Configure(LED_YELLOW);
    LED_Configure(LED_GREEN);
    LED_Clear(LED_YELLOW);
    LED_Clear(LED_GREEN);

    // Initialisierung aller Portleitungen des SPI-Ports und des SD-/MMC-Kartenslots
    InitAllPios();

    // Wenn whrend der Initialisierung eine SD-Karte gefunden wird...
    sdCardDetected = SD_CardDetect();
    if (sdCardDetected)
    {
        LED_Set(LED_YELLOW);
        LED_Clear(LED_GREEN);
    }

    // Initialisierung des Bitfeldes fuer die Karteninformationen
    SD_ClearBitfield(&sdInfo);

    while (1)
    {
        // In jedem Durchgang testen, ob eine SD-/MMC-Karte vorhanden ist.
        sdCardDetected = SD_CardDetect();

        // Keine SD-/MMC-Karte gefunden : Text einmal anzeigen
        if ((sdCardDetected == false) && (msgNoCardFound == false))
        {
            printf("Bitte SD-Karte einsetzen\r\n\r\n");
            LED_Clear(LED_YELLOW);
            LED_Clear(LED_GREEN);

            SD_ClearBitfield(&sdInfo);
            msgNoCardFound = true;
            msgCardInitStarted = false;
            msgCardInfoShown = false;
        }

        // SD-/MMC-Karte vorhanden : Testen, ob Karte bereits initialisiert wurde.
        if ((sdCardDetected == true)
                && (sdInfo.sdCardInitDone == !SD_CARD_INIT_FINISHED)
                && (msgCardInitStarted == false))
        {
            printf("Neue Karte gefunden --> Starte Initialisierung\r\n\r\n");
            LED_Set(LED_YELLOW);
            LED_Clear(LED_GREEN);

            SD_ClearBitfield(&sdInfo);
            msgNoCardFound = true;
            msgCardInitStarted = true;
            msgCardInfoShown = false;

            // Warte ca. 3 Sekunden, bevor Initialisierung gestartet wird.
            while (sdCardStartupTimeDone == false)
                ;

            // Starte Initialisierung
            sdInfo = SD_Init();
        }

        // SD-/MMC-Karte erkannt u. Initialisierung abgeschlossen --> Ergebnis pruefen
        // und einmal anzeigen
        if ((sdCardDetected == true)
                && (sdInfo.sdCardInitDone == SD_CARD_INIT_FINISHED)
                && (msgCardInfoShown == false))
        {
            printf(
                    "Initialisierung abgeschlossen --> Ergebnis ermitteln\r\n\r\n");

            // Standard-Evaluierung
            SD_EvalSdInfo(&sdInfo);

            // CID auswerten
            SD_GetCid(&cidInfo);

            // CSD auswerten
            SD_GetCsd(&csdInfo);

            msgCardInfoShown = true;


            result = f_mount(&fs, "", 1);
            FAT_EvalErrorCodes("f_mount() -->", result);


        }

        // ..., wenn es sich z.B. um eine MMC-Karte handelt
        if ((sdCardDetected == true)
                && (sdInfo.sdCardReady == !SD_CARD_READY_AND_USABLE))
        {
            if (sdCardNotUsable == true)
            {
                sdCardNotUsable = false;
                LED_Toggle(LED_GREEN);
            }
        }

        // Karte ist SD-Karte und laesst sich nutzen
        if ((sdCardDetected == true)
                && (sdInfo.sdCardReady == SD_CARD_READY_AND_USABLE))
        {
            LED_Set(LED_GREEN);
        }

        // Testen, ob Karte wieder entfernt wurde...
        if ((sdCardDetected == false)
                && (sdInfo.sdCardInitDone == SD_CARD_INIT_FINISHED))
        {
            LED_Clear(LED_YELLOW);
            LED_Clear(LED_GREEN);

            SD_ClearBitfield(&sdInfo);
            msgNoCardFound = false;
            msgCardInitStarted = false;
            msgCardInfoShown = false;
        }
    }
}

/**
 * void SysTick_Handler (void)
 *
 * \brief Systick im Millisekundentakt
 *
 * @param
 * @return  void
 */
void SysTick_Handler(void)
{
    timeStamp++;

    if ((timeStamp % 3000 == 0) && (sdCardDetected == true))
    {
        sdCardStartupTimeDone = true;
    }

    if (timeStamp % 500 == 0)
    {
        sdCardNotUsable = true;
    }
}

/**
 * void InitAllPios (void)
 *
 * \brief   Initialisiert alle PIO-Ports
 *
 * @param   none
 * @return  none
 */
static void InitAllPios(void)
{
    PIO_Configure(pinSD_CD, 1);
    PIO_Configure(pinSD_WP, 1);
    PIO_Configure(pinSD_CS, 1);
    PIO_Configure(pinSD_MISO, 1);
    PIO_Configure(pinSD_MOSI, 1);
    PIO_Configure(pinSD_SCK, 1);
}

// EOF
