/**
 * tcWave.c
 *
 *  Created on: 07.11.2013
 *      Author: ralf
 *
 * \brief
 */

#include "board.h"
#include "chip.h"
#include "global.h"
#include "tcWave.h"


static void TcWaveformConfigure(void);

const Pin pinTc0Tioa1 = PIN_TC0_TIOA1;

// Definition und Intialisierung des Arrays 'waveformConfigurations'
struct WaveformConfiguration waveformConfigurations[] =
{
    { TC_CMR_TCCLKS_TIMER_CLOCK4,  178, 30 },  // TC_CMR_TCCLKS_TIMER_CLOCK4 = BOARD_MCK / 128
    { TC_CMR_TCCLKS_TIMER_CLOCK3,  375, 50 },  // TC_CMR_TCCLKS_TIMER_CLOCK3 = BOARD_MCK /  32
    { TC_CMR_TCCLKS_TIMER_CLOCK3,  800, 75 },
    { TC_CMR_TCCLKS_TIMER_CLOCK2, 1000, 80 },  // TC_CMR_TCCLKS_TIMER_CLOCK2 = BOARD_MCK /   8
    { TC_CMR_TCCLKS_TIMER_CLOCK1, 4000, 55 }   // TC_CMR_TCCLKS_TIMER_CLOCK1 = BOARD_MCK /   2
};

uint8_t configuration = 0;
uint8_t numConfigurations = sizeof(waveformConfigurations) / sizeof(struct WaveformConfiguration);


/**
 * void TcWaveformInitialize (void)
 *
 * \brief Initialisiert TC0/Channel 1 als Generator
 *
 * @param   none
 * @return  void
 */
void TcWaveformInitialize(void)
{
    volatile uint32_t dummy;

    // Konfiguration von PIO-Port PA15 als Ausgang
    PIO_Configure(&pinTc0Tioa1, 1);

    // TC0 im PMC anmelden
    PMC_EnablePeripheral(ID_TC1);

    // Zunaechst werden der TC und seine Interrupts deaktiviert. Durch das Auslesen des Statusregisters
    // wird dieses auf einen definierten Wert zurckgesetzt
    REG_TC0_CCR1 = TC_CCR_CLKDIS;
    REG_TC0_IDR1 = 0xFFFFFFFF;
    dummy        = REG_TC0_SR1;

    // Konfiguration des Ausgangssignals, d.h. Auswahl der Frequenz und des Tastverhaeltnisses
    TcWaveformConfigure();

    // ... und zum Schluss TC0 wieder aktivieren
    REG_TC0_CCR1 = TC_CCR_CLKEN | TC_CCR_SWTRG;
}

/**
 * void TcWaveformConfigure (void)
 *
 * \brief   Konfiguration von TC0/Channel 1 als Rechteckgenerator
 *
 * @param   none
 * @return  void
 */
static void TcWaveformConfigure(void)
{
    const uint32_t divisors[5] = { 2, 8, 32, 128, BOARD_MCK / 32768 };
    uint32_t       prescaler;
    uint32_t       ra, rc;

    REG_TC0_CMR1 = waveformConfigurations[configuration].clockSelection |  // Auswahl der Taktquelle
                   TC_CMR_WAVE                                          |  // Wave-Mode aktivieren
                   TC_CMR_ACPA_SET                                      |  // RA-Wert erreicht --> TIOA1 = '1' setzen
                   TC_CMR_ACPC_CLEAR                                    |  // RC-Wert erreicht --> TIOA1 = '0' setzen
                   TC_CMR_CPCTRG;                                          // RC == Zaehler    --> Timer neu starten

    // rc nimmt den Wert auf, der der mit clockSelection gewaehlten Frequenz entspricht
    prescaler = divisors[waveformConfigurations[configuration].clockSelection];
    rc = BOARD_MCK / prescaler;
    rc /= waveformConfigurations[configuration].frequency;
    REG_TC0_RC1 = rc;

    // ra enthaelt den Wert, der dem Duty Cycle der gewuenschten Frequenz entspricht
    ra = (100 - waveformConfigurations[configuration].dutyCycle) * rc / 100;
    REG_TC0_RA1 = ra;
}


// EOF
