/**
 * tcCapture.c
 *
 *  Created on: 07.11.2013
 *      Author: ralf
 *
 * \brief
 */
#include <stdint.h>
#include "board.h"
#include "chip.h"
#include "tcCapture.h"
#include "global.h"

const Pin pinTc0Tioa2 = PIN_TC0_TIOA2;


/**
 * void TcCaptureInitialize (void)
 *
 * \brief Initialisierung von TC0/Channel 2 als Input (Capture-Modus)
 *
 * @param
 * @return  void
 */
void TcCaptureInitialize(void)
{
    volatile uint32_t dummy;

    PIO_Configure(&pinTc0Tioa2, 1);
    PMC_EnablePeripheral(ID_TC2);
    REG_TC0_CCR2 = TC_CCR_CLKDIS;
    REG_TC0_IDR2 = 0xFFFFFFFF;
    dummy        = REG_TC0_SR2;

    REG_TC0_CMR2 = TC_CMR_TCCLKS_TIMER_CLOCK2 |  // Channel 2 wird mit (BOARD_MCK / 8) getaktet
                   TC_CMR_LDRA_RISING         |  // Neuladen von RA bei steigender Flanke
                   TC_CMR_LDRB_FALLING        |  // Neuladen von RB bei fallender Flanke
                   TC_CMR_ABETRG              |  // Zeigt an, dass TIOA als Trigger verwendet wird
                   TC_CMR_ETRGEDG_FALLING;       // und dass die fallende Flanke genutzt werden soll

    // TC0/Channel 2 wird beim NVIC als Interruptquelle angemeldet
    NVIC_DisableIRQ(TC2_IRQn);
    NVIC_ClearPendingIRQ(TC2_IRQn);
    NVIC_SetPriority(TC2_IRQn, 0);
    NVIC_EnableIRQ(TC2_IRQn);
}

/**
 * void showCapturedPulses (void)
 *
 * \brief
 *
 * @param
 * @return  void
 */
void showCapturedPulses(void)
{
	uint16_t frequence;
	uint16_t dutyCycle;


    if (capturedPulses != 0)
    {
        REG_TC0_IDR2 = TC_IER_LDRBS;
        printf("Anzahl Impulse an TC0/Channel 2: %u,\r\n", (unsigned int) capturedPulses);
        printf("RA = %u\r\n", (unsigned int) capturedRA);
        printf("RB = %u\r\n", (unsigned int) capturedRB);

        frequence = (BOARD_MCK / 8) / capturedRB;
        dutyCycle = (capturedRB - capturedRA) * 100 / capturedRB;
        printf("Gemessene Frequenz = %d Hz\r\n", frequence);
        printf("Tastverhaeltnis    = %d%%\r\n", dutyCycle);

        capturedPulses = 0;
        capturedRA     = 0;
        capturedRB     = 0;
    }
    else
        printf("Keine Messung erfolgt!\r\n");

    printf("Druecken Sie die Taste 'h', um ins Hauptmenue zu gelangen\r\n");
}


/**
 * void TC2_IrqHandler (void)
 *
 * \brief Interrupt-Service-Routine fuer TC0 / Channel 2
 *
 * @param
 * @return  void
 */
void TC2_IrqHandler(void)
{
    uint32_t status;

    status = REG_TC0_SR2;
    if ((status & TC_SR_LDRBS) == TC_SR_LDRBS)
    {
        capturedPulses++;
        capturedRA = REG_TC0_RA2;
        capturedRB = REG_TC0_RB2;
    }
}

// EOF
