/**
 * main.c
 *
 *  Created on: 25.10.2013
 *      Author: ralf
 *
 * \brief
 */


static void InitAllPios(void);


// Allgemeine Header-Dateien
#include <stdio.h>
#include <stdbool.h>

// Spezielle Header-Dateien fr das Entwicklungsboard und den Chip
#include "board.h"
#include "board_olimex.h"
#include "chip.h"
#include "global.h"
#include "spi_main.h"
#include "sdcard.h"
#include "calc_crc.h"


// Diese sind in "board.h" definiert
const Pin pinSD_CD[]   = { PIN_SD_CD };
const Pin pinSD_WP[]   = { PIN_SD_WP };
const Pin pinSD_CS[]   = { PIN_SD_CS };
const Pin pinSD_SCK[]  = { PIN_SD_SCK };
const Pin pinSD_MOSI[] = { PIN_SD_MOSI };
const Pin pinSD_MISO[] = { PIN_SD_MISO };


uint32_t timeStamp;
bool     sdCardStartupTimeDone = false;

// SD-Karten-Kommandos
unsigned char CMD00[]    = { 0x40, 0x00, 0x00, 0x00, 0x00 };  // CMD0
unsigned char CMD08[]    = { 0x48, 0x00, 0x00, 0x01, 0xAA };  // CMD8
unsigned char CMD17[]    = { 0x57, 0x00, 0x00, 0x00, 0x00 };  // CMD17
unsigned char CMD41_V1[] = { 0x69, 0x00, 0x00, 0x00, 0x00 };  // CMD41 (V1.x-Version), HCS = '0'
unsigned char CMD41_V2[] = { 0x69, 0x40, 0x00, 0x00, 0x00 };  // CMD41 (V2.x-Version), HCS = '1'
unsigned char CMD55[]    = { 0x77, 0x00, 0x00, 0x00, 0x00 };  // CMD55
unsigned char CMD58[]    = { 0x7A, 0x00, 0x00, 0x00, 0x00 };  // CMD58
unsigned char CMD_DUMMY  = 0xFF;


SDINFO_t sdInfo;
bool     sdCardDetected  = false;

/**
 * M A I N - Funktion
 */
int main (void)
{
    bool msgNoCardFound     = false;
    bool msgCardInitStarted = false;


    // Watchdog-Timer deaktivieren
    WDT_Disable(WDT);

    // Erzeugung eines 1ms-Taktes aus dem Systemtakt
    if (SysTick_Config(BOARD_MCK / 1000))
    {
        return -1;
    }

    LED_Configure(LED_YELLOW);
    LED_Configure(LED_GREEN);
    LED_Clear(LED_YELLOW);
    LED_Clear(LED_GREEN);

    // Initialisierung aller Portleitungen des SPI-Ports und des SD-/MMC-Kartenslots
    InitAllPios();

    // Initialisierung des Bitfeldes fuer die Karteninformationen
    SD_ClearBitfield(&sdInfo);

    while (1)
    {
    	sdCardDetected = SD_CardDetect();

    	// Wenn keine SD-/MMC-Karte erkannt wurde, 1x diesen Text anzeigen
        if ((sdCardDetected == false) && (msgNoCardFound == false))
        {
            printf("Bitte SD-Karte einsetzen\r\n");
            LED_Clear(LED_YELLOW);
            LED_Clear(LED_GREEN);
            msgNoCardFound     = true;
            msgCardInitStarted = false;
        }

        // Wenn eine SD-/MMC-Karte erkannt wurde, die noch nicht initialisiert wurde
        if ((sdCardDetected == true) &&
        	(sdInfo.sdCardInitDone == !SD_CARD_INIT_FINISHED) &&
        	(msgCardInitStarted == false))
        {
            printf("Neue SD-Karte gefunden --> Starte Initialisierung\r\n");
            LED_Set(LED_YELLOW);
            msgCardInitStarted = true;

            // Warte drei Sekunden nach der Erkennung einer neuen SD-Karte
            while (sdCardStartupTimeDone == false)
                ;

            sdInfo = SD_Init();
            if (sdInfo.sdCardInitDone == SD_CARD_INIT_FINISHED)
            {
                printf("Initialisierung abgeschlossen\r\n");
                SD_EvalSdInfo(&sdInfo);
                LED_Set(LED_GREEN);
            }
            else
            {
                printf("Initialisierung nicht abgeschlossen\r\n");
            }
        }

        // SD-Karte wurde nach erfolgter Initialisierung wieder entfernt
        if ((sdCardDetected == false) && (sdInfo.sdCardInitDone == SD_CARD_INIT_FINISHED))
		{
            printf("SD-Karte entfernt\r\n");
            LED_Clear(LED_YELLOW);
            LED_Clear(LED_GREEN);

            SD_ClearBitfield(&sdInfo);

            msgNoCardFound        = false;
            msgCardInitStarted    = false;
		}
    }

    return -1;
}

/**
 * void SysTick_Handler (void)
 *
 * \brief Systick im Millisekundentakt
 *
 * @param
 * @return  void
 */
void SysTick_Handler(void)
{
    timeStamp++;

    if ((timeStamp % 3000 == 0) && (sdCardDetected == true))
    {
        sdCardStartupTimeDone = true;
    }
}


/**
 * void InitAllPios (void)
 *
 * \brief   Initialisiert alle PIO-Ports
 *
 * @param   none
 * @return  none
 */
static void InitAllPios(void)
{
	PIO_Configure(pinSD_CD, 1);
	PIO_Configure(pinSD_WP, 1);
	PIO_Configure(pinSD_CS, 1);
	PIO_Configure(pinSD_MISO, 1);
	PIO_Configure(pinSD_MOSI, 1);
	PIO_Configure(pinSD_SCK, 1);
}


// EOF
