/**
 * pulsewidthmod.c
 *
 *  Created on: 14.11.2013
 *      Author: ralf
 *
 * \brief
 */

#include "pulsewidthmod.h"
#include "board.h"
#include "board_olimex.h"

/**
 * Verwendet die Ports PA0 und PA30 zur Ansteuerung von zwei LEDs.
 * Die Ports sind Channel #0 des PWMC zugeordnet und zueinander
 * komplementaer.
 */
static const Pin pins[] =
{
    PIN_PWMC_PWMH0_PA0A,
    PIN_PWMC_PWML0_PA30A
};


void PWM_Initialize(void)
{
    uint32_t i;

    // Konfiguration von PWMH0 (PA0) und PWML0 (PA30)
    PIO_Configure(pins, PIO_LISTSIZE(pins));

    // PWM beim PMC registrieren
    PMC_EnablePeripheral(ID_PWM);

    // Verwendung von Clock A mit dem Wert 'PWM_FREQUENCY * MAX_DUTY_CYCLE' (= 2500).
    // Clock B wird nicht verwendet.
    PWMC_ConfigureClocks(PWM_FREQUENCY * MAX_DUTY_CYCLE, 0, BOARD_MCK);

    // Konfiguration des PWM-Kanals #0 fuer beide LEDs (linksbuendig,
    // Dead-time-Generator aktiviert
    PWMC_ConfigureChannelExt(PWM, CHANNEL_PWM_LED0, // Kanal #0
                             PWM_CMR_CPRE_CKA,      // Vorteiler
                             0,                     // Ausrichtung: linksbuendig
                             0,                     // Polaritaet
                             0,                     // countEventSelect
                             PWM_CMR_DTE,           // Dead-time enable
                             0,                     // Dead-time High invertertiert
                             0);                    // Dead-time Low invertiert

    PWMC_SetPeriod(PWM, CHANNEL_PWM_LED0, MAX_DUTY_CYCLE);
    PWMC_SetDutyCycle(PWM, CHANNEL_PWM_LED0, MIN_DUTY_CYCLE);
    PWMC_SetDeadTime(PWM, CHANNEL_PWM_LED0, DEADTIME_HIGH, DEADTIME_LOW);

    // Festlegung der synchronisierten Kanle, Update-Modus = 2: Update erfolgt nach
    // ablauf der Periode
    PWMC_ConfigureSyncChannel(PWM,
                              (1 << CHANNEL_PWM_LED0) | (1 << CHANNEL_PWM_LED1) | (1 << CHANNEL_PWM_LED2),
                              PWM_SCM_UPDM_MODE2,   //  (PWMC) Autom. Schreiben der Daten und autom.
                              0, 0);                //  Ausloesung (Trigger) des Updates

    // Einstellung der Periodendauer, nach der der Sync-Kanal aktualisiert wird
    PWMC_SetSyncChannelUpdatePeriod(PWM, PWM_SCUP_UPR(0xF));

    // Interrupt fuer den PDC-Transfer konfigurieren
    NVIC_DisableIRQ(PWM_IRQn);
    NVIC_ClearPendingIRQ(PWM_IRQn);
    NVIC_SetPriority(PWM_IRQn, 0);
    NVIC_EnableIRQ(PWM_IRQn);
    PWMC_EnableIt(PWM, 0, PWM_IER2_ENDTX);

    // Einstellung des Override-Wertes = High-Level
    PWMC_SetOverrideValue(PWM, PWM_OOV_OOVH0);


    // Duty-Cycle-Bbuffer fuellen
    for (i = 0; i < DUTY_BUFFER_LENGTH / 2; i++)
    {
        dutyBuffer[i] = (i + MIN_DUTY_CYCLE);  // Schreibt Werte von 0 bis 50
    }

    for (i = 0; i < DUTY_BUFFER_LENGTH / 2; i++)
    {
        dutyBuffer[i + DUTY_BUFFER_LENGTH / 2] = MAX_DUTY_CYCLE - i;    // Schreibt Werte von 50 - 0
    }

    // PDC-Transfer definieren ueber Startadresse des Buffers und seiner Laenge
    PWMC_WriteBuffer(PWM, dutyBuffer, DUTY_BUFFER_LENGTH);

    // PWM-Kanal #0 aktivieren
    PWMC_EnableChannel(PWM, CHANNEL_PWM_LED0);
}


/**
 * \brief Interrupt-Service-Routine fuer den PWM-Controller.
 */
void PWM_IrqHandler(void)
{
	uint32_t isr2 = PWM->PWM_ISR2;

	if ((isr2 & PWM_ISR2_ENDTX) == PWM_ISR2_ENDTX)
	{

		PWMC_WriteBuffer(PWM, dutyBuffer, DUTY_BUFFER_LENGTH);
	}
}

// EOF pulsewidthmod.c
