/**
 * Datei: rtt_timer.c
 * Autor: Ralf Jesse
 * Datum: 02.09.2013
 *
 * \brief
 *
 */


/**
 * Include-Dateien
 */
#include <stdio.h>
#include <stdbool.h>

#include "board.h"
#include "chip.h"
#include "global.h"


/**
 * Variablen
 */
volatile uint8_t timer_1ms     = 0;
volatile uint8_t timer_10ms    = 0;
volatile uint8_t timer_100ms   = 0;
volatile uint8_t timer_1000ms  = 0;
volatile uint8_t timer_10000ms = 0;

volatile bool     alarm;
volatile uint8_t  alarm_timer;
volatile uint8_t  alarm_blink_timer;


/**
 * void Configure_RTT(void)
 *
 * \brief Konfiguration des Real-time Timers RTT
 *
 * @param  none
 * @return none
 */
void Configure_RTT(void)
{
    uint32_t prevTime;

    // Annahme:
    // Laut Datenblatt hat Slow Clock SLCK eine Frequenz von 32,768 kHz,
    // d.h. mit einem Vorteilerwert (Prescaler) von 32768 kann der Timer
    // pro Sekunde bis 32768 zaehlen.
    //
    // ACHTUNG:
    // SLCK betrgt nur dann genau 32,768 kHz, wenn ein externer Oszillator
    // verwendet wird, der beim Olimex-Board nicht vorhanden ist. In diesem
    // Fall kann SLCK zwischen 22 und 42 kHz liegen (RTT wird vom Supply
    // Controller (SUPC) versorgt): Daher kann man bei diesem Board nicht
    // davon ausgehen, dass ein Vorteiler von 32768 einen exakten 1-Sekunden-
    // Takt erzeugt!
    RTT_SetPrescaler(RTT, 28);        // Prescaler = 28 --> ca. 1-ms-Takt
    RTT_SetAlarm(RTT, 28000);         // Alarm-Timer 28000 --> ca. 1-s-Takt

    prevTime = RTT_GetTime(RTT);
    while (prevTime == RTT_GetTime(RTT))  // Lt. Datenblatt soll der Timer
    	;                                 // mind. 2x gelesen werden

    // Zunchst wird der RTT als Interruptquelle deaktiviert sowie alle noch
    // ausstehenden Interruptanforderungen des RTT gelscht. Dann wird die
    // Prioritt festgelegt. Nun muss der RTT wieder als Interrupt-Quelle im
    // NVIC angemeldet werden. Der letzte Schritt ist die Aktivierung des
    // RTT-Interrupts.
    NVIC_DisableIRQ(RTT_IRQn);
    NVIC_ClearPendingIRQ(RTT_IRQn);
    NVIC_SetPriority(RTT_IRQn, 0);
    NVIC_EnableIRQ(RTT_IRQn);
    RTT_EnableIT(RTT, RTT_MR_RTTINCIEN | RTT_MR_ALMIEN);
}


/**
 * void RTT_IrqHandler(void)
 *
 * \brief Interrupt-Service-Routine (ISR). Diese Funktion wird automatisch
 *        aufgerufen, wenn der RTT einen Interrupt auslst. Der Zeiger auf
 *        diese Funktion ist in der Vektortabelle definiert.
 *
 * @param  none
 * @return none
 */
void RTT_IrqHandler(void)
{
    uint32_t status;

    // Status-Registers auslesen, setzt Interrupt-Flags zurck
    status = RTT_GetStatus(RTT);

    // Wurde das Timer-Inkrement-Bit gesetzt?
    if ((status & RTT_SR_RTTINC) == RTT_SR_RTTINC)
    {
        // Erzeugt 1-Millisekunden-Tick
        if ((timer_1ms++ > 0) && (timer_1ms++ % 10 == 0))
        {
            timer_1ms = 0;
            timer_10ms++;
        }

        // Erzeugt 10-Millisekunden-Tick
        if ((timer_10ms > 0) && (timer_10ms % 10 == 0))
        {
            timer_10ms = 0;
            timer_100ms++;
        }

        // Erzeugt 100-Millisekunden-Tick
        if ((timer_100ms > 0) && (timer_100ms % 10 == 0))
        {
            timer_100ms = 0;
            timer_1000ms++;

            timer_LedGruen++;
            timer_LedGelb++;
            alarm_timer++;
            alarm_blink_timer++;
        }

        // Alles zurcksetzen nach 10 Sekunden
        if ((timer_1000ms > 0) && (timer_1000ms %10 == 0))
        {
            timer_1ms     = 0;
            timer_10ms    = 0;
            timer_100ms   = 0;
            timer_1000ms  = 0;
        }
    }

    // Hat der Timer einen Alarm ausgelst?
    if ((status & RTT_SR_ALMS) == RTT_SR_ALMS)
    {
        alarm = true;
        NVIC_DisableIRQ(RTT_IRQn);
        RTT_SetPrescaler(RTT, 28);
        RTT_SetAlarm(RTT, 28000);
        NVIC_EnableIRQ(RTT_IRQn);
        RTT_EnableIT(RTT, RTT_MR_RTTINCIEN | RTT_MR_ALMIEN);
    }
}


// EOF
