/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support
 * ----------------------------------------------------------------------------
 * Copyright (c) 2009, Atmel Corporation
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

/**
 *  \page getting_started Getting Started with SAM3S Microcontrollers
 *
 *  \section Purpose
 *
 *  The Getting Started example will help new users get familiar with Atmel's
 *  SAM3S family of microcontrollers. This basic application shows the startup
 *  sequence of a chip and how to use its core peripherals.
 *
 *  \section Requirements
 *
 *  This package can be used with SAM3S evaluation kits.
 *
 *  \section Description
 *
 *  The demonstration program makes two LEDs on the board blink at a fixed rate.
 *  This rate is generated by using Time tick timer. The blinking can be stopped
 *  using two buttons (one for each LED).
 *
 *  \section Usage
 *
 *  -# Build the program and download it inside the evaluation board. Please
 *     refer to the
 *     <a href="http://www.atmel.com/dyn/resources/prod_documents/doc6224.pdf">
 *     SAM-BA User Guide</a>, the
 *     <a href="http://www.atmel.com/dyn/resources/prod_documents/doc6310.pdf">
 *     GNU-Based Software Development</a>
 *     application note or to the
 *     <a href="ftp://ftp.iar.se/WWWfiles/arm/Guides/EWARM_UserGuide.ENU.pdf">
 *     IAR EWARM User Guide</a>,
 *     depending on your chosen solution.
 *  -# On the computer, open and configure a terminal application
 *     (e.g. HyperTerminal on Microsoft Windows) with these settings:
 *    - 115200 bauds
 *    - 8 bits of data
 *    - No parity
 *    - 1 stop bit
 *    - No flow control
 *  -# Start the application.
 *  -# Two LEDs should start blinking on the board. In the terminal window, the
 *     following text should appear (values depend on the board and chip used):
 *     \code
 *      -- Getting Started Example xxx --
 *      -- xxxxxx-xx
 *      -- Compiled: xxx xx xxxx xx:xx:xx --
 *     \endcode
 *  -# Pressing and release button 1 should make the first LED stop & restart
 *     blinking; pressing button 2 should make the other LED stop & restart
 *     blinking.
 *
 *  \section References
 *  - getting-started/main.c
 *  - pio.h
 *  - pio_it.h
 *  - led.h
 *  - trace.h
 */

/** \file
 *
 *  This file contains all the specific code for the getting-started example.
 *
 */

/**
 *        Headers
 */
#include "board.h"

#include <stdbool.h>
#include <stdio.h>


/**
 * Konstanten / Constants
 */
#define IRQ_PRIOR_PIO	0
#define BLINK_PERIOD	1000


/**
 * Externe Referenzen / External references
 */
extern void _ConfigureTc(void);
extern void _ConfigureLeds(void);
extern void _ConfigureButtons(void);
extern void _Wait(unsigned long);


extern bool bLed0Active;

/*----------------------------------------------------------------------------
 *        Exported functions
 *----------------------------------------------------------------------------*/

/**
 *  \brief getting-started Application entry point.
 *
 *  \return Unused (ANSI-C compatibility).
 */
//extern int main(void)
int main(void)
{

#if defined   ( __CC_ARM   ) /* Keil Vision 4 */
	/* Disable semihosting */
#    pragma import(__use_no_semihosting_swi)
#endif

	/* Disable watchdog */
	WDT_Disable(WDT );

	/* Output example information */
	printf("-- Getting Started Example %s --\n\r", SOFTPACK_VERSION);
	printf("-- %s\n\r", BOARD_NAME);
	printf("-- Compiled: %s %s --\n\r", __DATE__, __TIME__);

	/* Configure systick for 1 ms. */
	printf("Configure system tick to get 1ms tick period.\n\r");
	if (SysTick_Config(BOARD_MCK / 1000))
	{
		printf("-F- Systick configuration error\n\r");
	}

	/* PIO configuration for LEDs and Buttons. */
	PIO_InitializeInterrupts(IRQ_PRIOR_PIO);

	printf("Configure TC.\n\r");
	_ConfigureTc();

	printf("Configure LED PIOs.\n\r");
	_ConfigureLeds();

	printf("Configure buttons with debouncing.\n\r");
	_ConfigureButtons();

	printf("Press USRBP1 to Start/Stop the blue LED D2 blinking.\n\r");
	printf("Press USRBP2 to Start/Stop the green LED D3 blinking.\n\r");

	while (1)
	{
		/* Wait for LED to be active */
		while (!bLed0Active)
			;

		/* Toggle LED state if active */
		if (bLed0Active)
		{
			LED_Toggle(0);
			printf("0 ");
		}

		/* Wait for 500ms */
		_Wait(500);
	}
}

// EOF
