/**
 * sdcard.c
 *
 *  Created on: 29.12.2013
 *      Author: ralf
 *
 * \brief
 */

#include <stdint.h>

#include "board.h"
#include "global.h"
#include "spi_main.h"
#include "sdcard.h"
#include "calc_crc.h"


/**
 * unsigned char Init_SD (void)
 *
 * \brief   Initialisierung der SD-Karte. Nach einem Power-on-Reset
 *          befinden sich SD-Karten im SD-Protokollmodus. Aus Kosten-
 *          gruenden kann dieser hier nicht genutzt werden. Daher
 *          erfolgt zunaechst der Wechsel in den SPI-Modus.
 *
 * @param
 * @return  struct SDINFO_t (--> sdcard.h)
 */
//unsigned char SD_Init(void)
SDINFO_t SD_Init(void)
{
    SDINFO_t sdInfo;

    uint8_t  count;
    uint32_t r1Response;
    uint32_t r3r7Response;
    uint32_t sdCardVersion = !SD_CARD_VERSION_2_TYPE;
    uint32_t tryCmd41Count = 0;


    //
    //----- Noch ist das SD-Protokoll aktiv -----
    //

    SPI_Init();                                             // Initialisierung des SPI

    SD_SetChipSelect(CS_HIGH);                              // Mind. 74 Taktzyklen warten mit Chip Select = CS_HIGH
    for (count = 0; count < 10; count++)
       SPI_SendByte(CMD_DUMMY);                             // Dummy-Kommando erzeugt Taktsignal

    SD_SendCommand(CMD00);                                  // CMD0 --> Wechsel vom SD- in das SPI-Protokoll
    r1Response = SD_GetR1Response();                        // Antwort der SD-Karte auslesen
    SD_SetChipSelect(CS_HIGH);

    if (r1Response != SD_CARD_IDLE_STATE)                   // Auswertung der Antwort der SD-Karte
        sdInfo.sdIdleState = !SD_CARD_IDLE;   // Timeout, oder Initialisierung noch nicht beendet
    else
        sdInfo.sdIdleState = SD_CARD_IDLE;                  // Karte ist bereit (Leerlauf)

    //
    //----- Ab hier wird die SD-Karte im SPI-Modus betrieben -----
    //

    if (sdInfo.sdIdleState == SD_CARD_IDLE)
    {
        SD_SendCommand(CMD08);                              // CMD8 --> Auslesen der Interface-Bedingungen
        r1Response = SD_GetR1Response();                    // CMD8 --> Wird nur von Spec. 2.0 unterstuetzt.
        SD_SetChipSelect(CS_HIGH);                          //          Spec. 1.0-Karten --> ILLEGAL_COMMAND

        sdCardVersion = SD_CheckSpecVersion(r1Response);    // Spezifikation 1.x oder 2.x ?
        if (sdCardVersion == SD_CARD_VERSION_2_TYPE)
        {
    	    sdInfo.sdSpecVersion = SD_CARD_VERSION_2_TYPE;  // Karte entspricht Spezifikation V2.0

    	    r3r7Response = SD_GetR3R7Response();    	    // Rest der Antwort lesen und auswerten
            if (!(r3r7Response == (SD_VOLTAGE_27_TO_36VOLT | CMD08[4])))
    	    {
    	        sdInfo.sdVoltRangeAndPattern = !SD_CARD_VOLTRANGE_AND_PATTERN_OK;
    	        SD_SetChipSelect(CS_HIGH);
    	    }
    	    else
    	    {
    	        sdInfo.sdVoltRangeAndPattern = SD_CARD_VOLTRANGE_AND_PATTERN_OK;
    	    }
        }
        else
    	    sdInfo.sdSpecVersion = !SD_CARD_VERSION_2_TYPE;  // Karte entspricht Spezifikation V1.0

        if (((sdInfo.sdSpecVersion == SD_CARD_VERSION_2_TYPE)                    &&
        	 (sdInfo.sdVoltRangeAndPattern == SD_CARD_VOLTRANGE_AND_PATTERN_OK)) ||
             (sdInfo.sdSpecVersion == !SD_CARD_VERSION_2_TYPE))
        {
            SD_SendCommand(CMD58);                          // CMD58 -- > OCR auslesen
            r1Response = SD_GetR1Response();

            if (r1Response & SD_CARD_ILLEGAL_COMMAND)
            {
                sdInfo.sdCardType = !SD_CARD_IS_SD_TYPE;    // Karte ist MMC-Karte oder nicht nutzbar
                SD_SetChipSelect(CS_HIGH);
            }
            else
                sdInfo.sdCardType = SD_CARD_IS_SD_TYPE;     // Kann V1.0 oder V2.0 sein!
        }

        if (sdInfo.sdCardType == SD_CARD_IS_SD_TYPE)
        {
            // Karte ist nutzbar --> Auswertung des OCR-Registers
            r3r7Response = SD_GetR3R7Response();
            SD_SetChipSelect(CS_HIGH);

            // Bereich der Versorgungsspannung pruefen: Bits 20/21 muessen '1' sein
            if (!(r3r7Response & SD_CARD_VOLTRANGE_VALID))
                sdInfo.sdVoltRangeOk = !SD_CARD_VOLTRANGE_OK;    // Karte nicht nutzbar
            else
                sdInfo.sdVoltRangeOk = SD_CARD_VOLTRANGE_OK;        // Karte ist nutzbar
        }

        // CMD41 muss so lange gesendet werden, bis das Idle-Bit der Response '0' ist.
        // Hier wird die Anzahl der Versuche auf SD_CARD_MAX_TRY_SEND_CMD41 (= 32000)
        // gesetzt.

        if ((sdInfo.sdCardType == SD_CARD_IS_SD_TYPE && (sdInfo.sdVoltRangeOk == SD_CARD_VOLTRANGE_OK)))
        {
            for (tryCmd41Count = 0; tryCmd41Count < SD_CARD_MAX_TRY_SEND_CMD41; tryCmd41Count++)
            {
                // Vor dem Senden von CMD41 wird CMD55 gesendet
                SD_SendCommand(CMD55);                          // CMD55 --> Zeigt der SD-Karte an, dass das naechste Kommando
                r1Response = SD_GetR1Response();                //           applikationsspezifisch ist.
                SD_SetChipSelect(CS_HIGH);

                if (r1Response != 0xFF)                         // Fehler, wenn in der Antwort alle Bits = '1' sind
                {
                    if (sdCardVersion == SD_CARD_VERSION_2_TYPE)
                        SD_SendCommand(CMD41_V2);               // CMD41 fuer V2.0-Karten (HCS-Bit = '1')
                    else
                        SD_SendCommand(CMD41_V1);               // CMD41 fuer V1.0-Karten (HCS-Bit = '0')
                }

                r1Response = SD_GetR1Response();
                SD_SetChipSelect(CS_HIGH);

                if (r1Response & SD_CARD_ILLEGAL_COMMAND)       // Auswertung der Antwort von CMD41
                    sdInfo.sdCardType = !SD_CARD_IS_SD_TYPE;

                if ((r1Response & SD_CARD_IDLE_STATE) == 0)
                {
                    sdInfo.sdCardInitDone = SD_CARD_INIT_FINISHED;
                    break;
                }
            }
        }
        SD_SetChipSelect(CS_HIGH);

        // Wenn die SD-Karte Spezifikation 2.x (oder neuer) entspricht
        // --> OCR auslesen (Prueft das CCS-Bit --> Kartenkapazitaet High oder Low)
        if (sdInfo.sdCardInitDone == SD_CARD_INIT_FINISHED)
        {
            if (sdCardVersion == SD_CARD_VERSION_2_TYPE)
            {
                SD_SendCommand(CMD58);
                r1Response   = SD_GetR1Response();      // Kann ignoriert werden, muss aber vorhanden sein!
                r3r7Response = SD_GetR3R7Response();
                SD_SetChipSelect(CS_HIGH);

                if (r3r7Response & SD_CARD_HIGH_CAP_SUPPORT)
                    sdInfo.sdCapInfo = SD_CARD_HIGH_CAP_TYPE;
                else
                    sdInfo.sdCapInfo = !SD_CARD_HIGH_CAP_TYPE;
            }
            else
                sdInfo.sdCapInfo = !SD_CARD_HIGH_CAP_TYPE;
        }
        sdInfo.sdCardReady = SD_CARD_READY_AND_USABLE;
    }

    SPI->SPI_CSR |= (3 << 8);    // Einstellung der Taktfrequenz fuer den Datentransfer nach Abschluss der
                                 // Initialisierung. Die Taktfrequenz errechnet sich aus dem Quotienten von
                                 // BOARD_MCK und dem hier eingestellten Wert (--> SCBR) --> ca. 21MHz.
    return sdInfo;
}


/**
 * uint32_t SD_CheckSpecVersion (uint32_t)
 *
 * \brief   Prueft, ob die erkannte SD-/MMC-Karte Spezifikation V1.x oder V2.x
 *          entspricht. Die Pruefung erfolgt durch Auswertung von Bit 2
 *          (SD_CARD_ILLEGAL_COMMAND) von r1Response:
 *
 *          '0' --> Karte entspricht V2.x
 *          '1' --> Karte entspricht V1.x
 *
 * @param
 * @return  uint32_t
 */
uint32_t SD_CheckSpecVersion(uint32_t r1Response)
{
    uint32_t sdCardSpecVersion = !SD_CARD_VERSION_2_TYPE;

    if (!(r1Response & SD_CARD_ILLEGAL_COMMAND))
        sdCardSpecVersion = SD_CARD_VERSION_2_TYPE;

    return sdCardSpecVersion;
}


/**
 * void SD_SetChipSelect (uint8_t state)
 *
 * \brief  Aendert den Zustabd der CS-Leitung (Chip Select) entsprechend
 *         dem angegebenen Argument. Zulaessige Werte sind CS_HIGH und
 *         CS_LOW.
 *
 * @param   uint8_t : state = Neuer Status der CS-Leitung
 * @return  void
 */
void SD_SetChipSelect(uint8_t state)
{
    if (state == CS_LOW)
        PIO_Clear(pinSD_CS);    // !!! Invertierte Logik !!!
    else
    {
        PIO_Set(pinSD_CS);      // !!! Invertierte Logik !!!
        SPI_SendByte(0xFF);
    }
}


/**
 * uint32_t SD_SendCommand (unsigned char *)
 *
 * \brief
 *
 * @param
 * @return  uint32_t
 */
uint32_t SD_SendCommand(unsigned char *command)
{
    uint8_t  cmdLen;
    uint8_t  crc7 = 0;
    uint8_t  *pCmd = (uint8_t *)command;

    SD_SetChipSelect(CS_LOW);                    // Setze CS auf Low-Pegel

    for (cmdLen = 0; cmdLen < 5; cmdLen++)
        SPI_SendByte(*(command++));              // Sendet das Byte

    // Pruefsumme berechnen und ebenfalls senden
    crc7 = calc_crc7(pCmd, 5);                   // CRC7 berechnen ...
    SPI_SendByte(crc7);                          // ... und senden

    return 0;
}


/**
 * uint8_t SD_GetR1Response (void)
 *
 * \brief
 *
 * @param
 * @return  uint8_t
 */
uint8_t SD_GetR1Response(void)
{
    uint8_t count;
    uint8_t response;

    // Max. 10 Versuche, eine Antwort der SD-Karte zu lesen
    for (count = 0; count < 10; count++) {
        response = SPI_GetByte();
        if (response != 0xFF)
            break;
    }

    return response;
}


/**
 * uint32_t SD_GetR3R7Response (void)
 *
 * \brief   Antwort der SD-Karte entsprechend Format R3 oder R7.
 *          Die Antwort besteht aus fuenf Byte, wobei das erste
 *          Byte der R1-Response entspricht. Daher muss zunaechst
 *          immer SD_GetR1Response() ausgefuehrt werden!
 *
 * @param
 * @return  uint32_t
 */
uint32_t SD_GetR3R7Response(void)
{
    uint32_t response;

    response = SPI_GetByte()  << 24;
    response |= SPI_GetByte() << 16;
    response |= SPI_GetByte() << 8;
    response |= SPI_GetByte();

    return response;
}


/**
 * bool SD_CardDetect (void)
 *
 * \brief   Testet, ob eine SD-Karte gefunden wurde.
 *          true  --> SD-/MMC-Karte gefunden
 *          false --> SD-/MMC-Karte nicht gefunden
 *
 * @param   none
 * @return  bool
 */
//void SD_CardDetect(SDINFO_t *pSdInfo)
bool SD_CardDetect(void)
{
    if (PIO_Get(pinSD_CD) == 0)
    	return true;
    else
    	return false;
}


/**
 * bool SD_WriteProtect (void)
 *
 * \brief   Prueft, ob der Schreibschutz der SD-Karte aktiv ist.
 *
 * @param   none
 * @return  bool
 */
bool SD_WriteProtect(void)
{
    return (PIO_Get(pinSD_WP) == 0);
}


/**
 * void SD_ClearBitfield (void)
 *
 * \brief   Initialisiert das Bitfeld der Karteninformation.
 *          Setzt alle Bits auf '0' zurueck.
 *
 * @param
 * @return  void
 */
void SD_ClearBitfield(SDINFO_t *pSdInfo)
{
    pSdInfo->sdIdleState           = !SD_CARD_IDLE;
    pSdInfo->sdVoltRangeAndPattern = !SD_CARD_VOLTRANGE_AND_PATTERN_OK;
    pSdInfo->sdVoltRangeOk         = !SD_CARD_VOLTRANGE_OK;
    pSdInfo->sdCardInitDone        = !SD_CARD_INIT_FINISHED;
    pSdInfo->sdCardReady           = !SD_CARD_READY_AND_USABLE;
    pSdInfo->sdSpecVersion         = !SD_CARD_VERSION_2_TYPE;
    pSdInfo->sdCapInfo             = !SD_CARD_HIGH_CAP_TYPE;
    pSdInfo->sdCardType            = !SD_CARD_IS_SD_TYPE;
    pSdInfo->sdWriteProtectState   = SD_WriteProtect();              // ACHTUNG: Funktionsaufruf
    pSdInfo->sdDummy1              = 0;
    pSdInfo->sdDummy2              = 0;
    pSdInfo->sdDummy3              = 0;
    pSdInfo->sdDummy4              = 0;
    pSdInfo->sdDummy5              = 0;
    pSdInfo->sdDummy6              = 0;
}

/**
 * void SD_EvalSdInfo (SDINFO_t*)
 *
 * \brief   Auswertung von SDINFO_t sdInfo
 *
 * @param   SDINFO_t *pSdInfo
 * @return  void
 */
void SD_EvalSdInfo(SDINFO_t *pSdInfo)
{
    printf("\r\n");
    if (pSdInfo->sdIdleState == !SD_CARD_IDLE)
        printf("Eval: SD-/MMC-Karte Timeout oder noch nicht initialisiert\r\n");

    if (pSdInfo->sdCardType == SD_CARD_IS_SD_TYPE)
    {
        printf("Eval: Karte ist SD-Karte\r\n");

        if (pSdInfo->sdSpecVersion == SD_CARD_VERSION_2_TYPE)
            printf("Eval: SD-Karte entspricht Spezifikation 2.0\r\n");
        else
            printf("Eval: SD-Karte entspricht Spezifikation 1.0\r\n");

        if (pSdInfo->sdCapInfo == SD_CARD_HIGH_CAP_TYPE)
            printf ("Eval: SD-Karte ist ein High-Capacity-Type\r\n");
        else
            printf ("Eval: SD-Karte ist ein Low-Capacity-Type\r\n");
    }
    else
        printf("Eval: Karte ist MMC-Karte oder nicht nutzbar\r\n");
}
