/**
 * a2d.c : Analog-to-Digital
 *
 *  Created on: 01.12.2013
 *      Author: ralf
 *
 * \brief
 */
#include "board.h"
#include "global.h"
#include "a2d.h"

/**
 * void ADC_Configure (void)
 *
 * \brief
 *
 * @param
 * @return  void
 */
void ADC_Configure(Adc *pAdc, uint32_t channels[], uint32_t numChannels, uint32_t startup, uint32_t prescale)
{
    uint8_t i;

	// Initialisierung des ADC
	ADC_Initialize(pAdc, ID_ADC);

	// Einstellung des ADC-Taktes:
	//     ADCClock = MCK / ((PRESCALE+1) * 2)
	ADC_cfgFrequency(ADC, startup, prescale);
	ADC_CheckFrequency(ADC, BOARD_MCK);

	// Aktivieren der ausgwaehlten Kanaele
    for(i = 0; i < numChannels; i++)
    {
        ADC_EnableChannel(ADC, channels[i]);

        // Beim Olimex-Board ist der Temperatursensor an Kanal AD5 angeschlossen.
        // Dieser Test muss fuer andere Umgebungen evtl. angepasst werden.
        if (channels[i] == ADC_CHANNEL_5)
            ADC_EnableTS(ADC, ADC_ACR_TSON);
    }

	/* Enable ADC interrupt */
	NVIC_EnableIRQ(ADC_IRQn);

	/* Start conversion */
	ADC_StartConversion(ADC);

	ADC_ReadBuffer(ADC, adc_values, BUFFER_SIZE);

	/* Enable PDC channel interrupt */
	ADC_EnableIt(ADC, ADC_IER_RXBUFF);

}

/**
 * void ADC_ConfigureSequence (Adc*, uint8_t[], uint8_t)
 *
 * \brief   Konfiguriert die Sequenzregister des ADC in Abhaengigkeit von der
 *          Zahl der Kanaele, die erfasst werden sollen.
 *
 * @param   Adc*      : Pointer auf die ADC-Instanz
 * @param   uint8_t[] : Array mit den Nummern der Kanaele, die erfasst werden sollen
 * @param   uint8_t   : Anzahl der Kanaele
 * @return  void
 */
void ADC_ConfigureSequence(Adc *pAdc, uint32_t channels[], uint32_t numChannels, uint32_t startup, uint32_t prescale)
{
    uint8_t i;

    ADC_Initialize(ADC, ID_ADC);

	// Einstellung des ADC-Taktes:
	//     ADCClock = MCK / ((PRESCALE+1) * 2)
	ADC_cfgFrequency(ADC, startup, prescale);
	ADC_CheckFrequency(ADC, BOARD_MCK);

	// Aktivieren der ausgwaehlten Kanaele
    for(i = 0; i < numChannels; i++)
    {
        ADC_EnableChannel(ADC, channels[i]);

        // Beim Olimex-Board ist der Temperatursensor an Kanal AD5 angeschlossen.
        // Dieser Test muss fuer andere Umgebungen evtl. angepasst werden.
        if (channels[i] == ADC_CHANNEL_5)
            ADC_EnableTS(ADC, ADC_ACR_TSON);
    }

    // Festlegung der Reihenfolge der Sequenz, in der die verschiedenen
    // AD-Eingaenge erfasst werden.
    for (i = 0; i < numChannels; i++)
    {
        if (i < 8)
            pAdc->ADC_SEQR1 |= channels[i] << (4 * i);        // Kanaele < 8 --> ADC_SEQR1
        else
            pAdc->ADC_SEQR2 |= channels[i] << (4 * (i-8));    // Kanaele > 8 --> ADC_SEQR2
    }

	/* Enable ADC interrupt */
	NVIC_EnableIRQ(ADC_IRQn);

	/* Start conversion */
	ADC_StartConversion(ADC);

	ADC_ReadBuffer(ADC, adc_values, BUFFER_SIZE);

	/* Enable PDC channel interrupt */
	ADC_EnableIt(ADC, ADC_IER_RXBUFF);
}

/**
 * void ADC_IrqHandler (void)
 *
 * \brief
 *
 * @param
 * @return  void
 */
void ADC_IrqHandler(void)
{
	uint32_t status;

	status = ADC_GetStatus(ADC);

	if ((status & ADC_ISR_RXBUFF) == ADC_ISR_RXBUFF)
		ADC_ReadBuffer(ADC, adc_values, BUFFER_SIZE);
}

/**
 * void ADC_CheckConversionStatus()
 *
 * \brief
 *
 * @param
 * @return  void
 */
void ADC_CheckConversionStatus()
{
	uint32_t status;

	status = ADC_GetStatus(ADC);

	// Test: Konvertierung abgeschlossen?
	if ((status & ADC_ISR_EOC4) == ADC_ISR_EOC4)
	{
	    voltageDone = true;
        ADC_StartConversion(ADC);
	}

	if ((status & ADC_ISR_EOC5) == ADC_ISR_EOC5)
	{
	    temperatureDone = true;
	    ADC_StartConversion(ADC);
	}
}


/**
 * void ADC_CheckFrequency (Adc*, uint32_t)
 *
 * \brief   ADC_CheckFrequency() ist nur eine Wrapper-Funktion fuer die
 *          von Atmel bereitgestellte Funktion ADC_check(). Der Name der
 *          Originalfunktion ist meiner Meinung nach nicht gut gewaehlt.
 *
 * @param   Adc*     : Pointer auf die ADC-Instanz
 * @param   uint32_t : BOARD_MCK (bei Olimex = 64MHz)
 * @return  void
 */
void ADC_CheckFrequency(Adc *pAdc, uint32_t mckFrequency)
{
    ADC_check(ADC, BOARD_MCK);
}


// EOF (a2d.c)
