/**
 * systick.c
 *
 *  Created on: 26.11.2013
 *      Author: ralf
 *
 * \brief
 */

#include <stdbool.h>

#include "board.h"
#include "chip.h"
#include "global.h"

volatile bool     systickToggle;
volatile bool     dacc_eoc_event;
volatile uint32_t dwTimeStamp;


/**
 * void SysTick_Handler (void)
 *
 * \brief Interrupt-Service-Routine fuer den SysTick-Takt.
 *
 * @param
 * @return  void
 */
void SysTick_Handler(void)
{
    uint32_t daccIrqStatus;

    dwTimeStamp++;
    systickToggle = true;

    // Abfrage des DACC-Interrupt-Status-Registers
    daccIrqStatus = DACC_GetStatus(DACC);

    tableIndex++;

    if (tableIndex >= SAMPLES)
        tableIndex = 0;

    // Nur aufrufen, wenn EOC = 1 (EOC = End of conversion)
    if ((daccIrqStatus & DACC_ISR_EOC) == DACC_ISR_EOC)
    {
        if (waveForm == SINUS)
            DACC_SetConversionData(DACC, sign * sinusData[tableIndex] * amplitude / (MAX_DIGITAL / 2) + MAX_DIGITAL / 2);
        else if (waveForm == TRIANGLE)
              DACC_SetConversionData(DACC, sign * triangleData[tableIndex] * amplitude / (MAX_DIGITAL / 2));
        else if (waveForm == SAWTOOTH)
            DACC_SetConversionData(DACC, sign * sawtoothData[tableIndex] * amplitude / (MAX_DIGITAL / 2));
    }
}


/**
 * void delayMikros(uint32_t delay)
 *
 * \brief Der Wert 'dwTimeStamp' wird alle 10s dekrementiert, kuerzere
 *        Perioden als 10s lassen sich mit unseren Moeglichkeiten mess-
 *        technisch nicht nachweisen oder bestaetigen. Es lassen sich
 *        daher nur Vielfache von 10s einstellen!
 *
 *        !!! A C H T U N G !!!
 *        Der Prescaler fuer BOARD_MCK wird hierfuer in der main()-Funktion
 *        auf 100.000 eingestellt.
 *
 * @param  uint32_t delay : Vielfaches von 10s
 * @return none
 */
void delayMikros(uint32_t delay)
{
    volatile uint32_t start = dwTimeStamp;
    uint32_t          elapsed;

    // Korrigiert 'delay' so, dass ein 'delay'-Wert von 10 zu einem Systick
    // von 10s fhrt. Auf diese Weise lassen sich aber nur Vielfache von
    // 10s einstellen.
    delay /= 10;
    do
    {
        elapsed = dwTimeStamp;
        elapsed -= start;
    }
    while (elapsed < delay);
}


/**
 * void delayMillis (uint32_t)
 *
 * \brief Der Wert 'dwTimeStamp' wird alle 10s dekrementiert, kuerzere
 *        Perioden als 10s lassen sich mit unseren Moeglichkeiten mess-
 *        technisch nicht nachweisen oder bestaetigen. Durch einen
 *        Korrekturfaktor lassen sich aber Vielfache von 1ms einstellen.
 *
 *        !!! A C H T U N G !!!
 *        Der Prescaler fuer BOARD_MCK wird hierfuer in der main()-Funktion
 *        auf 100.000 eingestellt.
 *
 * @param
 * @return  void
 */
void delayMillis(uint32_t delay)
{
    volatile uint32_t start = dwTimeStamp;
    uint32_t          elapsed;

    // Korrekturfaktor (10s * 100 * delay)
    delay *= 100;
    do
    {
        elapsed = dwTimeStamp;
        elapsed -= start;
    }
    while (elapsed < delay);
}


/**
 * void delaySeconds (uint32_t)
 *
 * \brief Der Wert 'dwTimeStamp' wird alle 10s dekrementiert, kuerzere
 *        Perioden als 10s lassen sich mit unseren Moeglichkeiten mess-
 *        technisch nicht nachweisen oder bestaetigen. Durch einen
 *        Korrekturfaktor lassen sich aber Vielfache von 1ms einstellen.
 *
 *        !!! A C H T U N G !!!
 *        Der Prescaler fuer BOARD_MCK wird hierfuer in der main()-Funktion
 *        auf 100.000 eingestellt.
 *
 * @param
 * @return  void
 */
void delaySeconds(uint32_t delay)
{
    volatile uint32_t start = dwTimeStamp;
    uint32_t          elapsed;

    // Korrekturfaktor (100000 * delay) = 'delay' Sekunden
    delay *= 100000;
    do
    {
        elapsed = dwTimeStamp;
        elapsed -= start;
    }
    while (elapsed < delay);
}

// EOF (systick.c)
