/**
 * table(.exe)
 * ===========
 *
 * Datum / Date:   Nov. 24, 2013
 * Autor / Author: Ralf Jesse
 *
 * table(.exe) berechnet eine Tabelle, die am Analog-Ausgang des DACC der
 * AT91SAM3S-Mikrocontrollerfamilie ein Signal mit der angegebenen Wellen-
 * form erzeugt. Die Aufloesung des Signals verbessert sich mit der Anzahl
 * der Elemente. table(.exe) erzeugt eine Textdatei mit dem durch den Parameter
 * 'form' anagegeben Namen --> {form}.txt. Diese Datei befindet sich nach der
 * Ausfuehrung des Programms im gleichen Verzeichnis, in dem sich table(.exe)
 * befindet. Zur Verbesserung der Lesbarkeit der erzeugten Tabellen werden pro
 * Zeile nur 10 Hexadezimalwerte eingetragen, nach jeweils 5 Zeilen wird eine
 * Leerzeile eingefuegt.
 *
 * table(.exe) calculates a table which generates an analog signal for the
 * analog output of the AT91SAM3S's DACC. The number of 'elements' defines
 * the quality of the analog output signal. table(.exe) creates a .txt file of
 * the form {waveform}.txt in the same folder from which the programm was
 * run. To improve the readability of the generated table, each row has 10
 * (hexadecimal) comma-separated values. After five rows an additional blank
 * line is inserted.
 *
 * Anwendung / Usage: table(.exe) waveform elements
 *
 * Parameter:
 * ----------
 * waveform : sin/SIN, tri/TRI, saw/SAW
 * elements : Numerischer Wert. Muss ein Vielfaches von 10 sein.
 *            Number. Must be a multiple of 10.
 **/

#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <ctype.h>

#include "global.h"


// Globale Variable
FILE *fp = NULL;
bool roundDown = false; // Abrunden, wenn Anzahl der Elemente kein Viel-
int  tableSize;         // Groesse der Tabelle (muss Vielfaches von 10 sein)
int  waveForm;
bool tableHeaderDone = false;

int main(int argc, char **argv)
{
    // Variablen deklarieren
    char *name; // Name der Zieldatei

    // Anzahl der Argumente ueberpruefen (muss inkl. des Programmnamens 3 sein!
    if (argc != 3)
    {
        usage(ERR_NO_OF_ARGS);
        return -1;
    }

    // Pruefen, ob zulaessige Signalform gewaehlt wurde
    name = stringToLowerCase(argv[1]);
    if (((strcmp(name, "sin") != 0) &&
            (strcmp(name, "tri") != 0) &&
            (strcmp(name, "saw") != 0)
            ))
    {
        usage(ERR_WRONG_WAVEFORM);
        return -1;
    }

    // Umwandlung von Parameter 2 in Zahlenwert
    tableSize = stringToNumber(argv[2]);
    if (tableSize < 0)
    {
        usage(ERR_INV_CHARS);
        return -1;
    }

    // Hinweis, dass Anzahl der Elemente <> Vielfaches von 10 --> Abrundung
    if (roundDown == true)
    {
        roundDown = false;
        printf("\r\n\r\n");
        printf("Anzahl der Elemente wurde abgerundet auf naechstkleineres ");
        printf("Vielfaches von 10. Neue Tabellengroesse: %d Elemente.\r\n",
                tableSize);
    }

    // Festlegung der gewaehlten Signalform
    if (strcmp(name, "sin") == 0)
        waveForm = SINUS;
    if (strcmp(name, "tri") == 0)
        waveForm = TRIANGLE;
    if (strcmp(name, "saw") == 0)
        waveForm = SAWTOOTH;

    // Name der Zieldatei aus Argument 'wave' erzeugen
    name = strcat(argv[1], ".txt");

    // Datei oeffnen (bzw. erstellen)
    fp = fopen(name, "w+");
    if (fp == NULL)
    {
        printf("Konnte Datei %s nicht erstellen\r\n", name);
        return -1;
    }

    // Aufruf des Tabellengenerators entsprechend Param #1
    switch (waveForm)
    {
        case SINUS:
            calcSinusData();
            break;

        case TRIANGLE:
            calcTriangleData();
            break;

        case SAWTOOTH:
            calcSawtoothData();
            break;
    }

    // Datei muss am Ende geschlossen werden!
    fclose(fp);

    return 0;
}


/**
 * void usage(int errorNumber)
 *
 * \brief Zeigt die Nutzung des Programmes bei falscher Anwendung mit
 *        entsprechender Fehlermeldung an.
 *
 * @param errorNumber
 */
void usage(int errorNumber)
{
    switch (errorNumber)
    {
        case ERR_NO_OF_ARGS:
            printf("Fehler: Falsche Anzahl Programmargumente\r\n\r\n");
            break;

        case ERR_WRONG_WAVEFORM:
            printf("Fehler: Unzulaessiger Signalname\r\n\r\n");
            break;

        case ERR_INV_CHARS:
            printf("Fehler: Param #2 enthaelt unzulaessige Zeichen\r\n\r\n");
            break;

        case ERR_NOT_MULTIPLE_OF_10:
            printf("Fehler: Param #2 kein Vielfaches von 10! Abgerundet auf ");
            printf("naechstkleineren moeglichen Wert\r\n\r\n");
            break;
    }
    printf("Anwendung / Usage: table(.exe) waveform elements\r\n");
    printf("waveform := {sin, tri, saw}\r\n");
    printf("elements := Vielfaches von 10 / Multiple of 10\r\n");
}

