/*----------------------------------------------------------------------------
 *        Headers
 *----------------------------------------------------------------------------*/

#include "board.h"
#include "board_olimex.h"
#include "global.h"
#include "pulsewidthmod.h"

#include <stdint.h>
#include <stdio.h>


/**
 * \brief Startpunkt fuer das "PWM mit PDC"-Beispiel.
 *
 * Das Projekt steuert zwei LEDs an, die an den zueinander komplementaeren
 * PWM-Ausgaengen PWMH0 (PA0, Peripheriegeraet A) und PWML0 (PA30, Peripheriegeraet A)
 * angeschlossen sind. Das Ausgangssignal an PWMH0 startet beim Maximalwert und wird
 * dann stufenlos bis auf 0 heruntergefahren, bei PWML0 ist das Verhalten entsprechend
 * invertiert.
 *
 * Outputs a PWM on LED1 & LED2 & LED3 to makes it fade in repeatedly.
 * Channel #0, #1, #2 are linked together as synchronous channels, so they have
 * the same source clock, the same period, the same alignment and
 * are started together. The update of the duty cycle values is made
 * automatically by the Peripheral DMA Controller (PDC).
 *
 * \return Unused (ANSI-C compatibility).
 */
int main(void)
{
    uint8_t key;
    int32_t numkey;

    /* Disable watchdog */
    WDT_Disable( WDT);

    /* Startmeldung anzeigen */
    ShowIntro();

    /* PIO configuration */
    PWM_Initialize();


    while (1)
    {
        ShowMenu();
        key = UART_GetChar();

        switch (key)
        {
            case 'u':
                printf("Eingabe: Update-Periode %d bis %d.\r\n", 0, PWM_SCUP_UPR_Msk);
                numkey = GetNumkey2Digit();

                if (numkey <= PWM_SCUP_UPR_Msk)
                {
                    // Update der Periodendauer des Sync-Channels einstellen
                    PWMC_SetSyncChannelUpdatePeriod(PWM, numkey);
                    printf("Fertig\n\r");
                }
                else
                {
                    printf("Unzulaessige Eingabe\r\n");
                }
                break;

            case 'd':
                printf("Eingabe: Dead-time fuer Kanal #0 zwischen %d und %d\r\n", MIN_DUTY_CYCLE, MAX_DUTY_CYCLE);
                numkey = GetNumkey2Digit();

                if (numkey >= MIN_DUTY_CYCLE && numkey <= MAX_DUTY_CYCLE)
                {
                    // Update der Dead-time des Sync-Channels einstellen
                    PWMC_SetDeadTime(PWM, CHANNEL_PWM_LED0, numkey, numkey);

                    // Sync-Channel-Aktualisierung: Sperre aufheben
                    PWMC_SetSyncChannelUpdateUnlock(PWM);
                    printf("Fertig\n\r");
                }
                else
                {
                    printf("Unzulaessige Eingabe\r\n");
                }
                break;

            case 'o':
                printf("0: Kanal #0 --> Override fuer H0 --> 0\r\n");
                printf("1: Kanal #0 --> Override fuer H0 --> 1\r\n");
                printf("x: Kanal #0 --> PWM-Betrieb\r\n");
                key = UART_GetChar();

                if (key == '0')
                {
                    PWMC_EnableOverrideOutput(PWM, PWM_OSSUPD_OSSUPH0 | PWM_OSSUPD_OSSUPL0, 1);
                    PWMC_SetOverrideValue(PWM, ~PWM_OOV_OOVH0);
                    printf("Fertig\r\n");
                }
                else if (key == '1')
                {
                    PWMC_EnableOverrideOutput(PWM, PWM_OSSUPD_OSSUPH0 | PWM_OSSUPD_OSSUPL0, 1);
                    PWMC_SetOverrideValue(PWM, PWM_OOV_OOVH0);
                    printf("Fertig\r\n");
                }
                else if (key == 'x')
                {
                    PWMC_DisableOverrideOutput(PWM, PWM_OSSUPD_OSSUPH0 | PWM_OSSUPD_OSSUPL0, 1);
                    PWMC_DisableOverrideOutput(PWM, PWM_OOV_OOVH0 | PWM_OOV_OOVL0, 1);
                    printf("Fertig\r\n");
                }
                break;

            default:
                printf("Unzulaessige Eingabe\r\n");
                break;
        }
    }
}

