/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support
 * ----------------------------------------------------------------------------
 * Copyright (c) 2010, Atmel Corporation
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

/**
 * Dieses Beispiel ist aus Sicht des Ergebnisses und der Bedienung eine 1:1-Adaption
 * des Beispielprojekts, das von Atmel im Rahmen des >>Softpack for CodeSourcery<<
 * zur freien Verfgung ausgeliefert wird. Aus Gruenden der Uebersichtlichkeit habe
 * ich das Projekt aber auf mehrere Dateien aufgesplittet, sodass Sie die wesentlichen
 * Bestandteile in Ihren eigenen Projekten wiederverwenden koennen. Kommentare habe ich
 * ins Deutsche uebersetzt, ebenso die Texte, die zur Steuerung im Terminal-Emulator
 * angezeigt werden.
 */

/**
 * \page Projekt 11_01_TIMER_COUNTER
 *
 * \section Einsatz
 *
 * Dieses Beispiel zeigt eine moegliche Anwendung von TC (Timer/Counter). An Port PA15
 * werden rechteckfoermige Signale ausgegeben (Waveform-Modus), die dann an Port PA26
 * zurueckgefuehrt und dort gezaehlt werden.
 *
 * \section Voraussetungen
 *
 *    + SAM3S Evaluierungsboard von Atmel oder Olimex SAM3-P256
 *    + Direkte Verbindung zwischen Port PA15 (TC0, Channel 1, TIOA1) und Port PA26
 *      (TC0, Channel 2, TIOA2). TC0/Channel 1 ist im Waveform-Modus konfiguriert (Ausgang),
 *      TC0/Channel 2 im Capture-Modus (Eingang).
 *
 * \section Beschreibung
 *
 * Gezeigt wird die Konfiguration der Peripheriekomponente TC0 (Timer/Counter). Verwendet
 * werden zwei Kanaele von TC0: Kanal 1 wird im Waveform-Modus zur Erzeugung von Taktsignalen
 * konfiguriert, Kanal 2 dient als Eingang zur Zaehlung der mit Kanal 1 generierten Impulse.
 * Die Register RA und RB werden beim Auftreten eines TC-Interrupts neu geladen. Der Inhalt
 * von RA und RB liefert die Anzahl der gezaehlten Impulse. Die Auswahl der Taktrate und die
 * Anzeige des Ergebnisses erfolgt auf einem Terminal-Emulator.
 *
 * \section Zusammenfassung
 * <ul>
 * <li>Auswahl einer vordefinierten Frequenz und des Tastverhaeltnisses des erzeugten Signals.
 *
 *
 * <li>Configure TC channel 1 as waveform output.
 * <li>Configure TC channel 2 as capture input.
 * <li>Configure capture Register A be loaded when rising edge of TIOA occurs.
 * <li>Configure capture Register B be loaded when failing edge of TIOA occurs.
 * <li>Configure an interrupt for TC and enable the RB load interrupt.
 * <li> 'c' start capture.
 * <li> 's' will stop capture,and dump the informations what have been captured.
 * </ul>
 *
 * \section Usage
 *
 * -# Compile the application.
 * -# Connect the DBGU port of the evaluation board to the computer and open
 * it in a terminal.
 *    - Settings: 115200 bauds, 8 bits, 1 stop bit, no parity, no flow control.
 * -# Download the program inside the evaluation board and run it. Please refer to
 *    <a href="http://www.atmel.com/dyn/resources/prod_documents/doc6224.pdf">
 *    SAM-BA User Guide</a>, the
 *    <a href="http://www.atmel.com/dyn/resources/prod_documents/doc6310.pdf">
 *    GNU-Based Software Development</a> application note or to the
 *    <a href="ftp://ftp.iar.se/WWWfiles/arm/Guides/EWARM_UserGuide.ENU.pdf">
 *    IAR EWARM User Guide</a>, depending on your chosen solution.
 * -# Upon startup, the application will output the following line on the DBGU:
 *    \code
 *     -- TC capture waveform example  xxx --
 *     -- xxxxxx-xx
 *     -- Compiled: xxx xx xxxx xx:xx:xx --
 *    \endcode
 * -# Choose the item in the following menu to test.
 *    \code
 *     Menu :
 *     ------
 *       Output waveform property:
 *       0: Set Frequency =  120 Hz, Duty Cycle = 30%
 *       1: Set Frequency =  375 Hz, Duty Cycle = 50%
 *       2: Set Frequency =  800 Hz, Duty Cycle = 75%
 *       3: Set Frequency = 1000 Hz, Duty Cycle = 80%
 *       4: Set Frequency = 4000 Hz, Duty Cycle = 55%
 *       -------------------------------------------
 *       c: Capture waveform from TC 0 channel 2
 *       s: Stop capture and display informations what have been captured
 *       h: Display menu
 *     ------
 *    \endcode
 *
 * \section References
 * - tc_capture_waveform/main.c
 */

/**
 * \file
 *
 * This file contains all the specific code for the tc capture waveform example.
 */

/*----------------------------------------------------------------------------
 *        Headers
 *----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdint.h>

#include "board.h"
#include "tcCapture.h"
#include "global.h"


int main(void)
{
    uint8_t  ucKey;

    /* Disable watchdog */
    WDT_Disable( WDT);

    TcWaveformInitialize();
	TcCaptureInitialize();

	/* Display menu */
	DisplayMenu();

	while (1)
	{
	    ucKey = UART_GetChar();

	    switch (ucKey)
	    {
	        case 'h':
	            DisplayMenu();
	            break;

	        case 's':
	            showCapturedPulses();
	            break;

	        case 'c':
	            printf("Messung starten. Druecken Sie 's' zum Beenden\r\n");
	            REG_TC0_IER2 = TC_IER_LDRBS;
	            REG_TC0_CCR2 = TC_CCR_CLKEN | TC_CCR_SWTRG;
	            break;

	        default:
	            if ((ucKey >= '0') && (ucKey < ('0' + numConfigurations)))
	            {
                    if (!capturedPulses)
                    {
                        configuration = ucKey - '0';
                        TcWaveformInitialize();
                    }
	            }
	            else
	            {
	                printf("Zaehlmodus aktiv. Druecken Sie die Taste 's' zum Beenden...\r\n");
	            }
	            break;
	    }
	}
}

// EOF
