/**
 * Datei: stateMachine.c
 * Autor: Ralf Jesse
 * Datum: 17.09.2013
 *
 * \brief
 *
 *  =================================================     =====================================================
 *  | Aufbau des Registers RTC_TIMR (Time Register) |     | Aufbau des Registers RTC_CALR (Kalender Register) |
 *  =================================================     =====================================================
 *  | Bit(s)  | Funktion                            |     | Bit(s)  | Funktion                                |
 *  |-----------------------------------------------|     |---------------------------------------------------|
 *  | 23 - 31 | reserviert                          |     | 30 - 31 | reserviert                              |
 *  |-----------------------------------------------|     |---------------------------------------------------|
 *  | 22      | Im 12h-Modus: AM = 0 / PM = 1       |     | 28 - 29 | Tag des Monats: 10er                    |
 *  |-----------------------------------------------|     | 24 - 27 | Tag des Monats: 1er                     |
 *  | 20 - 21 | Stunden: 10er                       |     |---------------------------------------------------|
 *  | 16 - 19 | Stunden: 1er                        |     | 21 - 23 | Wochentag                               |
 *  |-----------------------------------------------|     |---------------------------------------------------|
 *  | 15      | reserviert                          |     | 20      | Monat: 10er                             |
 *  |-----------------------------------------------|     | 16 - 19 | Monat:  1er                             |
 *  | 12 - 14 | Minuten: 10er                       |     |---------------------------------------------------|
 *  |  8 - 11 | Minuten: 1er                        |     | 12 - 15 | Jahr:  10er                             |
 *  |-----------------------------------------------|     |  8 - 11 | Jahr:   1er                             |
 *  |  7      | reserviert                          |     |---------------------------------------------------|
 *  |-----------------------------------------------|     |  7      | reserviert                              |
 *  |  4 -  6 | Sekunden: 10er                      |     |---------------------------------------------------|
 *  |  0 -  3 | Sekunden: 1er                       |     |  4 -  6 | Jahrhundert: 10er                       |
 *  =================================================     |  0 -  3 | Jahrhundert:  1er                       |
 *                                                        =====================================================
 */

/**
 * Include-Dateien
 */
#include <stdio.h>
#include <stdbool.h>

#include "board.h"
#include "chip.h"
#include "global.h"
#include "hd44780.h"
#include "stateMachine.h"

/**
 * Funktion: setNewState()
 *
 * \brief Statusmaschine zum Einstellen des RTC
 *        PB1: Statuswechsel
 *        PB2: bernimmt neuen Status, wenn vorher PB1 gedrckt wurde, sonst bleibt
 *             der alte Status erhalten.
 *
 * @param  none
 * @return uint8_t currentStatus
 */
uint8_t setNewState(void)
{
    static uint8_t tempState;
    static bool    tempRtcSetupStarted = false;

	// Statuswechsel: STATE_NORMAL_RUN --> STATE_SET_RTC
    if (currentState == STATE_NORMAL_RUN && (pb1Pressed == true || pb2Pressed == true))
    {
        if (stateModeChanged == false)
        {
            stateModeChanged = true;
            rtcSetupStarted  = true;

            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "RTC einst. (J/N)");
            HD44780_ClearLine(LCD_LINE_2);
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Ja");
        }

        // PB1: Toggelt die Auswahl --> RTC einstellen: Ja/Nein
        if (pb1Pressed == true)
        {
            pb1Pressed = false;

            if (tempRtcSetupStarted == true)
            {
                tempRtcSetupStarted = false;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Nein");
            }
            else
            {
                tempRtcSetupStarted = true;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Ja");
            }
        }

        // Mit PB2 --> Auswahl bernehmen
        if (pb2Pressed == true)
        {
            pb2Pressed       = false;
            stateModeChanged = false;

            if (tempRtcSetupStarted == true)
            {
                rtcSetupStarted = true;
                currentState    = STATE_SET_RTC;
            }
            else
            {
                rtcSetupStarted = false;
                currentState    = STATE_NORMAL_RUN;
            }

            tempRtcSetupStarted = false;
        }
    }

    // Statuswechsel: STATE_SET_RTC --> STATE_SET_CLOCK
    //                              --> STATE_SET_DATE
    if (currentState == STATE_SET_RTC)
    {
        if (stateModeChanged == false)
        {
            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Uhrzeit/Kalender");
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Uhrzeit");

            tempState        = STATE_SET_CLOCK;
            stateModeChanged = true;
        }

        if (pb1Pressed == true)
        {
            pb1Pressed = false;

            if (rtcMode == MODE_RTC_SET_CLOCK)
            {
                rtcMode   = MODE_RTC_SET_DATE;
                tempState = STATE_SET_DATE;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Kalender");
            }
            else
            {
                rtcMode   = MODE_RTC_SET_CLOCK;
                tempState = STATE_SET_CLOCK;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Uhrzeit");
            }
        }

        if (pb2Pressed == true)
        {
            pb2Pressed       = false;
            stateModeChanged = false;
            currentState     = tempState;
        }
    }

    // Statuswechsel: STATE_SET_CLOCK --> STATE_SET_CLOCK_TYPE
    if (currentState == STATE_SET_CLOCK)
    {
        if (stateModeChanged == false)
        {
            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Uhr/Wecker");
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Uhr");

            stateModeChanged = true;
        }

        if (pb1Pressed == true)
        {
            pb1Pressed = false;

            if (clockType == NORMAL_CLOCK)
            {
                clockType = ALARM_CLOCK;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Wecker");
            }
            else
            {
                clockType = NORMAL_CLOCK;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Uhr");
            }
        }

        if (pb2Pressed == true)
        {
            pb2Pressed     = false;
            stateModeChanged = false;
            currentState   = STATE_SET_CLOCK_TYPE;
        }
    }

    // Statuswechsel: STATE_SET_CLOCK_TYPE --> STATE_SET_CLOCK_MODE
    if (currentState == STATE_SET_CLOCK_TYPE)
    {
        if (stateModeChanged == false)
        {
            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "12/24 Stunden");
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "24 Stunden");

            stateModeChanged = true;
        }

        if (pb1Pressed == true)
        {
            pb1Pressed = false;

            if (clockMode == MODE_24H)
            {
                clockMode = MODE_12H;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "12 Stunden");
            }
            else
            {
            	clockMode = MODE_24H;
                HD44780_ClearLine(LCD_LINE_2);
            	HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "24 Stunden");
            }
        }

        if (pb2Pressed == true)
        {
            pb2Pressed     = false;
            stateModeChanged = false;

            if (clockType == NORMAL_CLOCK)
                currentState = STATE_SET_NORMAL_CLOCK;
            else
                currentState = STATE_SET_ALARM_CLOCK;
        }
    }

    // Statuswechsel: STATE_SET_NORMAL_CLOCK --> STATE_SET_HOURS
    //                STATE_SET_ALARM_CLOCK  --> STATE_SET_HOURS
    if (currentState == STATE_SET_NORMAL_CLOCK || currentState == STATE_SET_ALARM_CLOCK)
    {
        if (currentState == STATE_SET_NORMAL_CLOCK)
        {
            if (stateModeChanged == false)
            {
                HD44780_ClearDisplay();
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Uhr stellen");

                // Zum Stellen der Uhr --> letzte Uhrzeit als Basis verwenden
                RTC_GetTime(RTC, &clockHours, &clockMinutes, &clockSeconds);
                convertTimeToString(&clockHours, &clockMinutes, &clockSeconds);
                HD44780_ClearLine(LCD_LINE_3);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_3, currentTime);

                stateModeChanged        = true;
            }

            setNewTime(&clockHours, &clockMinutes, currentTime);

            if (rtcSetupStarted == false)
            {
                stateModeChanged = false;
                setClockTime(clockMode, clockHours, clockMinutes, clockSeconds);
            }
        }

        if (currentState == STATE_SET_ALARM_CLOCK)
        {
        	if (stateModeChanged == false)
        	{
                HD44780_ClearDisplay();
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Wecker stellen");

                stateModeChanged        = true;
        	}
            setNewTime(&alarmHours, &alarmMinutes, currentTime);

            if (rtcSetupStarted == false)
            {
                stateModeChanged = false;
                setAlarmTime(clockMode, alarmHours, alarmMinutes, alarmSeconds);
            }
        }
    }

    //
    //---------- EINSTELLUNG DER KALENDERFUNKTIONEN ----------
    //

    // Statuswechsel: STATE_SET_DATE --> STATE_SET_NORMAL_DATE, oder
    //                               --> STATE_SET_ALARM_DATE
    if (currentState == STATE_SET_DATE)
    {
        if (stateModeChanged == false)
        {
            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Datum/Alarmdatum");
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Datum");

            stateModeChanged = true;
        }

        if (pb1Pressed == true)
        {
            pb1Pressed = false;

            if (calenderType == NORMAL_DATE)
            {
                calenderType = ALARM_DATE;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Alarmdatum");
            }
            else
            {
                calenderType = NORMAL_DATE;
                HD44780_ClearLine(LCD_LINE_2);
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, "Datum");
            }
        }

        if (pb2Pressed == true)
        {
            pb2Pressed       = false;
            stateModeChanged = false;

            if (calenderType == NORMAL_DATE)
                currentState         = STATE_SET_NORMAL_DATE;
            else
                currentState = STATE_SET_ALARM_DATE;
        }
    }

    // Ausfhren: Datum einstellen
    if (currentState == STATE_SET_NORMAL_DATE)
    {
        if (stateModeChanged == false)
        {
            HD44780_ClearDisplay();
            HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Datum einstellen");
        }

        // Einstellen des Datums
        currentState = STATE_SET_CENTURY;
    }

    // Diese Funktion muss immer wieder neu aufgerufen werden
    setNewDate(&dateYear, &dateMonth, &dateDay, &dateDayOfWeek, dateYearLCD);

    if (rtcSetupStarted == false)
    {
        dateYear += 2000;
        setCalendarDate(dateYear, dateMonth, dateDay, dateDayOfWeek);
    }

    //
    if (rtcSetupStarted == false)
    {
        if (currentState == STATE_SET_ALARM_DATE)
        {
            if (stateModeChanged == false)
            {
                HD44780_ClearDisplay();
                HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_1, "Alarmdatum");
            }
        }

//        setNewAlarmDate(&alarmMonth, &alarmDay, alarmDate);
    }

    return currentState;
}

/**
 * Funktion: static void setNewTime(uint8_t *pHours, uint8_t *pMinutes, char *display)
 *
 * \brief Einstellung der Stunden und Minuten fr Uhrzeit und Wecker. Die Sekunden
 *        werden immer auf '00' gesetzt!
 *
 *        PB1: Inkrementiert die Stunden und Minuten
 *        PB2: bernahme der Stunden/Minuten und Wechsel nach --> STATE_NORMAL_RUN
 *
 * @param  uint8_t *pHours   : Ptr auf die Variablen clockHours/alarmHours
 * @param  uint8_t *pMinutes : Ptr auf die Variablen clockMinutes/alarmMInutes
 * @param  char    *display  : Ptr auf currentTime f. Anzeige auf LC-Display
 *
 * @return none
 */
static void setNewTime(uint8_t *pHours, uint8_t *pMinutes, char *pDisplay)
{
	static bool hoursDone = false;
	static bool minutesDone = false;

	uint8_t hoursTenner = 0;               // LCD: Stunden, Zehner
	uint8_t hoursOnes = 0;               // LCD: Stunden, Einser

	uint8_t minutesTenner = 0;
	uint8_t minutesOnes = 0;

	uint8_t secondsTenner = 0;
	uint8_t secondsOnes = 0;

	hours = *pHours;
	minutes = *pMinutes;
	seconds = 0;

	// Einstellung der Stunden
	if (hoursDone == false)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			hours++;
			(*pHours)++;

			if (clockMode == MODE_24H)
			{
				if (hours > 23)
				{
					hours = 0;
					*pHours = 0;
				}
			}
			else // if (clockMode == MODE_12H)
			{
				if ((hours > 12) && (ampmMode == MODE_AM))
				{
					hours = 0;
					*pHours = 0;
					ampmMode = MODE_PM;
				}

				if ((hours > 12) && (ampmMode == MODE_PM))
				{
					hours = 0;
					*pHours = 0;
					ampmMode = MODE_AM;
				}

				if (ampmMode == MODE_AM)
					pDisplay[9] = 'A';
				else
					pDisplay[9] = 'P';

				pDisplay[10] = 'M';
			}

			// Anzeige des LCD aktualisieren
			hoursTenner = hours / 10;
			hoursOnes = hours % 10;
			minutesTenner = minutes / 10;
			minutesOnes = minutes % 10;
			secondsTenner = seconds / 10;
			secondsOnes = seconds & 10;

			pDisplay[0] = hoursTenner + '0';
			pDisplay[1] = hoursOnes + '0';
			pDisplay[3] = minutesTenner + '0';
			pDisplay[4] = minutesOnes + '0';
			pDisplay[6] = secondsTenner + '0';
			pDisplay[7] = secondsOnes + '0';

			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_3, pDisplay);
		}

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			hoursDone = true;
		}
	}

	if (hoursDone == true && minutesDone == false)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			minutes++;
			(*pMinutes)++;
		}

		if (minutes > 59)
		{
			minutes = 0;
			*pMinutes = 0;
		}

		// Anzeige des LCD aktualisieren
		hoursTenner = hours / 10;
		hoursOnes = hours % 10;
		minutesTenner = minutes / 10;
		minutesOnes = minutes % 10;

		pDisplay[0] = hoursTenner + '0';
		pDisplay[1] = hoursOnes + '0';
		pDisplay[3] = minutesTenner + '0';
		pDisplay[4] = minutesOnes + '0';

		HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_3, pDisplay);

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			hoursDone = false;
			minutesDone = false;

			rtcSetupStarted = false; // Der RTC-Setup-Modus zum Einstellen der Uhrzeit
									 // wird beendet --> zurck zum Normalbetrieb
			currentState = STATE_NORMAL_RUN;

			HD44780_ClearDisplay();
		}
	}

	// Einstellung der Sekunden
	clockSeconds = 0;
	alarmSeconds = 0;
}

/**
 * Funktion: static void setDate(uint16_t *pYear, uint8_t *pMonth, uint8_t *pDay, uint8_t *pDayOfWeek, char *pDisplay)
 *
 * \brief Einstellung von Jahrhundert, Jahr, Monat, Wochentag und Tag des Monats
 *
 *        PB1: Auswahl von Jahrhundert, Jahr, Monat, Wochentag und Tag des Monats
 *        PB2: bernahme der eingestellten Werte und bergang in die nchste Einstellung
 *
 * @param  uint8_t *pYear      : Jahrhundert + Jahr
 * @param  uint8_t *pMonth     : Monat
 * @param  uint8_t *pDay       : Tag des Monats
 * @param  uint8_t *pDayOfWeek : Wochentag
 * @param  char    *display    : Ptr fr die Anzeige auf LC-Display
 *
 * @return none
 */
static void setNewDate(uint16_t *pYear, uint8_t *pMonth, uint8_t *pDay,
		uint8_t *pDayOfWeek, char *pDisplay)
{
	uint8_t maxDays;
	uint8_t i = 0;

	// ACHTUNG:
	// Bei unserem Board wurde beim Auslesen des Datums IMMER das Jahr 20xx ausgegeben.
	// Ein Datum im Jahr 19xx wurde demnach immer auf 20xx 'korrigiert'. Ob dies ein
	// Fehler in der RTC ist, ist unbekannt! Aus diesem Grund wurde der entsprechende
	// Programmcode auskommentiert und durch Setzen des Jahres auf 20xx festgelegt. 19xx
	// kann also nicht ausgewhlt werden!

	// Statuswechsel: Von STATE_SET_NORMAL_DATE ber STATE_SET_CENTURY --> STATE_SET_YEAR
	if (currentState == STATE_SET_CENTURY)
	{
		if (stateModeChanged == false)
		{
			stateModeChanged = true;

			HD44780_ClearDisplay();

			// Zum Stellen des Datums --> letztes Datum als Basis verwenden
			RTC_GetDate(RTC, &dateYear, &dateMonth, &dateDay, &dateDayOfWeek);
			convertDateToString(&dateYear, &dateMonth, &dateDay,
					&dateDayOfWeek);
			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, dateYearLCD);
			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_3, dateMonthLCD);
			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_4, dateDayLCD);
		}

		// Das Jahrhundert wird fest eingestellt und ist nicht whlbar
		dateCentury = CENTURY_21;
		currentState = STATE_SET_YEAR;
	}

	// Statuswechsel: STATE_SET_YEAR --> STATE_SET_MONTH
	if (currentState == STATE_SET_YEAR)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			dateYear++;
			if (dateYear > 99)
				dateYear = 0;

			dateYearLCD[9] = dateYear / 10 + '0';
			dateYearLCD[10] = dateYear % 10 + '0';
			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_2, dateYearLCD);
		}

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			currentState = STATE_SET_MONTH;
		}
	}

	// Statuswechsel: STATE_SET_MONTH --> STATE_SET_DAY_OF_WEEK
	if (currentState == STATE_SET_MONTH)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			dateMonth++;
			if (dateMonth > 12)
				dateMonth = 1;

			// Der Name des Monats ist vom Typ char[] und wird in das char[]
			// fr die Anzeige im LCD kopiert. Der Monatsname endet mit '\0',
			// '\0' darf nicht mit kopiert werden --> beim Zugriff auf den
			// Monatsnamen muss der Index daher um '1' verkleinert werden.
			for (i = 0; i < 4; i++)
				dateMonthLCD[7 + i] = *(months[dateMonth - 1] + i);

			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_3, dateMonthLCD);
		}

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			currentState = STATE_SET_DAY_OF_WEEK;
		}
	}

	// Statuswechsel: STATE_SET_DAY_OF_WEEK --> STATE_SET_DAY
	if (currentState == STATE_SET_DAY_OF_WEEK)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			dateDayOfWeek++;
			if (dateDayOfWeek > 8)
				dateDayOfWeek = 1;

			// Wie bei den Monaten endet der Wochentag (char[]) mit '\0': Dieser
			// Wert darf nicht kopiert werden --> Index muss verkleinert werden.
			for (i = 0; i < 2; i++)
				dateDayLCD[7 + i] = *(days[dateDayOfWeek - 1] + i);

			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_4, dateDayLCD);
		}

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			currentState = STATE_SET_DAY;
		}
	}

	// Statuswechsel: STATE_SET_DAY --> STATE_SET_CALENDAR_DONE
	if (currentState == STATE_SET_DAY)
	{
		if (pb1Pressed == true)
		{
			pb1Pressed = false;

			// Bestimmung der maximalen Tage pro Monat
			if (dateMonth == JANUARY || dateMonth == MARCH || dateMonth == MAY
					|| dateMonth == JULY || dateMonth == AUGUST
					|| dateMonth == OCTOBER || dateMonth == DECEMBER)
				maxDays = 31;
			else if (dateMonth == APRIL || dateMonth == JUNE
					|| dateMonth == SEPTEMPER || dateMonth == NOVEMBER)
				maxDays = 30;
			else
				maxDays = 29;    // Es bleibt nur noch der Februar...

			dateDay++;
			if (dateDay > maxDays)
				dateDay = 1;

			dateDayLCD[12] = dateDay / 10 + '0';
			dateDayLCD[13] = dateDay % 10 + '0';
			HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_4, dateDayLCD);
		}

		if (pb2Pressed == true)
		{
			pb2Pressed = false;
			currentState = STATE_NORMAL_RUN;
		}
	}
}

/**
 * Funktion: static void setNewAlarmDate(uint8_t *pMonth, uint8_t *pDay, char *pDisplay)
 *
 * \brief Einstellung von Monat und Tag des Alarmdatums.
 *
 *        PB1: Inkrementiert die Monate und Tage
 *        PB2: bernahme der Monate und Tage
 *
 * @param  uint8_t *pMonth   : Ptr auf die Variablen clockHours/alarmHours
 * @param  uint8_t *pDay     : Ptr auf die Variablen clockMinutes/alarmMInutes
 * @param  char    *pDisplay : Ptr auf currentTime f. Anzeige auf LC-Display
 *
 * @return none
 */
static void setNewAlarmDate(uint8_t *pMonth, uint8_t *pDay, char *pDisplay)
{
	HD44780_PrintText(LCD_TEXT_LEFT, LCD_LINE_4, "Alarmdatum");
}

// EOF
