#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

#include "hd44780.h"

/**
 * struct file_operations
 */
static struct file_operations fops =
{
    owner   : THIS_MODULE,
    open    : hd44780_init,
    release : hd44780_exit,
    write   : hd44780_write
};

/**
 * hd44780_init
 * ============
 * Diese Funktion wird mit der Funktion open() aus dem Userspace aufgerufen.
 */
static int __init hd44780_init (void)
{
    int err;


    if (alloc_chrdev_region(&hd44780_numdevs, 0, 1, DEVICE_NAME) < 0)
    {
        printk (KERN_ALERT "alloc_chrdev_region failed with MAJOR = %d\n", MAJOR(hd44780_numdevs));
        return -EIO;
    }

    hd44780_object = cdev_alloc();
    if (hd44780_object == NULL)
    {
        goto free_device_number;
    }

    hd44780_object->owner = THIS_MODULE;
    hd44780_object->ops   = &fops;

    if (cdev_add (hd44780_object, hd44780_numdevs, 1))
    {
        goto free_cdev;
    }

    hd44780_class = class_create(THIS_MODULE, DEVICE_NAME);

    if (IS_ERR(hd44780_class))
    {
        pr_err("%s: no udev support\n", DEVICE_NAME);
        goto free_cdev;
    }

    hd44780_dev = device_create(hd44780_class, NULL, hd44780_numdevs, NULL, "%s", DEVICE_NAME);
    dev_info (hd44780_dev, "hd44780_init_called\n");


    err = hd44780_req_single_gpio (HD44780_RS);
    if (err < 0)
    {
        return -HD44780_REQ_SINGLE_GPIO_FAILED;
    }

    err = hd44780_req_single_gpio (HD44780_RW);
    if (err < 0)
    {
        return -HD44780_REQ_SINGLE_GPIO_FAILED;
    }

    err = hd44780_req_single_gpio (HD44780_ENABLE);
    if (err < 0)
    {
        return -HD44780_REQ_SINGLE_GPIO_FAILED;
    }

    err = hd44780_req_multi_gpio(hd44780_data);
    if (err < 0)
    {
        return -HD44780_REQ_MULTI_GPIO_FAILED;
    }


    // Zunaechst wird der HD44780 initialisiert.
    err = hd44780_setup();
    if (err < 0)
    {
        return -HD44780_INIT_FAILED;
    }

    return 0;

free_cdev:
    kobject_put(&hd44780_object->kobj);

free_device_number:
    unregister_chrdev_region(hd44780_numdevs, 1);
    return -EIO;

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * hd44780_open
 * ============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn dort die
 * open()-Funktion verwendet wird.
 */
static int hd44780_open (struct inode *device_file, struct file *instance)
{
#if 0
    int err;
    int gpio_num;

    for (gpio_num = GPIO_OFFSET; gpio_num < GPIO_OFFSET + GPIOS_USED; gpio_num++)
    {
        // Anforderung der benoetigten GPIOs.
        err = gpio_request(gpio_num, "GPIO");
        if (err)
        {
            printk (KERN_ALERT "gpio_request() failed for GPIO_%d\n", gpio_num);
            return -1;
        }

        // Alle GPIOs werden als Ausgang genutzt.
        err = gpio_direction_output(gpio_num, 0);
        if (err)
        {
            printk (KERN_ALERT "gpio_direction_output() failed for GPIO_%d\n", gpio_num);
            return -1;
        }
    }
#endif //

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * hd44780_close
 * =============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn das
 * Userspace-Programm beendet wird.
 */
static int __exit hd44780_close (struct inode *device_file, struct file *instance)
{
    gpio_set_value(HD44780_RS, LOW);
    gpio_set_value(HD44780_RW, LOW);
    gpio_set_value(HD44780_ENABLE, LOW);
    gpio_set_value(HD44780_DB4, LOW);
    gpio_set_value(HD44780_DB5, LOW);
    gpio_set_value(HD44780_DB6, LOW);
    gpio_set_value(HD44780_DB7, LOW);

    gpio_free(HD44780_RS);
    gpio_free(HD44780_RW);
    gpio_free(HD44780_ENABLE);
    gpio_free(HD44780_DB4);
    gpio_free(HD44780_DB5);
    gpio_free(HD44780_DB6);
    gpio_free(HD44780_DB7);


    return 0;
}

/**
 * hd44780_write
 * =============
 * Diese Funktion wird aus dem Userspace aufgerufen, z.B. bei
 * echo "text" > /dev/treiber.
 */
static ssize_t hd44780_write (struct file *instance, const char __user *user,
                             size_t count, loff_t *offset)
{
    unsigned long not_copied, to_copy;
    uint16_t      val;              // Wert aus der Applikation im Userspace
                                    // dataType = 1 --> Steuerkommando fuer MAX7219

    to_copy = min (count, sizeof (val));
    not_copied = copy_from_user (&val, user, to_copy);

    // Hier findet die >>Action<< statt.


    return to_copy - not_copied;
}

/**
 * hd44780_req_single_gpio
 * =======================
 * Anforderung eines einzelnen GPIO-Ports.
 */
static int hd44780_req_single_gpio (int gpio)
{
    return 0;
}

/**
 * hd44780_req_multi_gpio
 * =======================
 * Anforderung eines Arrays von GPIO-Ports.
 */
static int hd44780_req_multi_gpio(int *gpios)
{
    return 0;
}


/**
 * Zusatzinformationen, teilweise vorgeschrieben.
 */
module_init (hd44780_init);
module_exit (hd44780_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR ("Ralf Jesse");
MODULE_DESCRIPTION("Kernelmodul zur Ansteuerung eines LCD-Controllers vom Typ HD-44780.");
