/**
 * Autor / Author: ralf
 * Datum / Date  : 01.05.2013
 *
 */

/**
 * Header-Dateien / Header files
 */
#include "board.h"
#include <stdbool.h>


/**
 * Funktions-Prototypen / Function prototypes
 */
void _ConfigureButtons(void);
static void _Button1_Handler(const Pin *pPin);
static void _Button2_Handler(const Pin *pPin);
static void ProcessButtonEvent(uint8_t ucButton);

// Push buttons
const Pin pinPB1 = PIN_PUSHBUTTON_1;
const Pin pinPB2 = PIN_PUSHBUTTON_2;

/**
 * Externe Referenzen / External references
 */
extern bool bLed0Active;
extern bool bLed1Active;


/**
 * \brief Konfiguration der Pushbuttons / Configure Pushbuttons
 *
 * Konfiguriert den PIO als Input und erzeugt einen Interrupt, wenn der Taster
 * gedrueckt oder wieder losgelassen wird.
 *
 * Configures the PIO as input and generates an interrupt when the button is
 * pressed or released.
 */
void _ConfigureButtons(void)
{
	// Konfiguriere PIOs als Input / Configure PIOs as input
	PIO_Configure(&pinPB1, 1);
	PIO_Configure(&pinPB2, 1);

	// Entprellen der Eingaenge / Adjust debounce filter parameters (10 Hz)
	PIO_SetDebounceFilter(&pinPB1, 10);
	PIO_SetDebounceFilter(&pinPB2, 10);

	// Initialisiert die PIO-Interrupt-Handler (definiert in 'board.h')
	// Initialize the PIO IRQ handler (defined in 'board.h)
	PIO_ConfigureIt(&pinPB1, _Button1_Handler);
	PIO_ConfigureIt(&pinPB2, _Button2_Handler);

	// Freigabe der PIO-Controller-Interrupts
	// Enable PIO controller IRQs
	NVIC_EnableIRQ((IRQn_Type) pinPB1.id);
	NVIC_EnableIRQ((IRQn_Type) pinPB2.id);

	// Aktivieren der PIO-Anschluss-Interrupts
	// Enable the PIO line IRQs
	PIO_EnableIt(&pinPB1);
	PIO_EnableIt(&pinPB2);
}


/**
 * \brief Behandlung Button 1: IRQ ansteigende Flanke /
 *        Handler for button 1: IRQ rising edge
 *
 * Behandelt den Prozess zum Wechseln des Zustands von LED #0.
 * Handles the process to change the state of LED #0
 */
static void _Button1_Handler(const Pin *pPin)
{
	if (pPin == &pinPB1)
	{
		ProcessButtonEvent(0);
	}
}

/**
 * \brief Behandlung Button 2: IRQ fallende Flanke /
 *        Handler for button 2: IRQ falling edge
 *
 * Behandelt den Prozess zum Wechseln des Zustands von LED #1.
 * Handles the process to change the state of LED #1
 */
static void _Button2_Handler(const Pin *pPin)
{
	if (pPin == &pinPB2)
	{
		ProcessButtonEvent(1);
	}
}


/**
 * \brief
 *
 *
 */
static void ProcessButtonEvent(uint8_t ucButton)
{
	if (ucButton == 0)
	{
		bLed0Active = !bLed0Active;

		if (!bLed0Active)
		{
			LED_Clear(0);
		}
	}
	else
	{
		bLed1Active = !bLed1Active;

		if (bLed1Active)
		{
			LED_Set(1);
			TC_Start(TC0, 0);
		}
		else
		{
			LED_Clear(1);
			TC_Stop(TC0, 0);
		}
	}
}

// EOF
